"""Настройки Аксиомы."""

from abc import ABC, abstractmethod
from enum import Enum
from pathlib import Path
from typing import (
    Any,
    Dict,
    Iterator,
    List,
    Mapping,
    MutableMapping,
    Optional,
    Tuple,
    TypeVar,
    Union,
    cast,
)

from axipy._internal._metaclass import _MappingMetaDocumentation, _MappingMetaExtended
from axipy._internal._shadow_instance_factory import _shadow_manager
from axipy.cpp_core_core import DefaultSettingsList, ShadowSettings
from axipy.gui.gui_class import gui_instance
from PySide2.QtGui import QColor

__all__: List[str] = [
    "DefaultSettings",
    "CurrentSettings",
    "AxiomaLanguage",
]


class _ShadowSettingsManager:
    """Управляет доступом к внутренним настройкам Аксиомы."""

    Value = TypeVar("Value")
    """Значение."""
    ShadowValue = TypeVar("ShadowValue")
    """Внутреннее значение."""

    _PATH_KEYS = (  # Ключи настроек, для которых необходимо преобразование в Path и обратно в str
        "LastSavePath",
        "LastOpenPath",
        "LastPathWorkspace",
        "DefaultPathCache",
        "PenCatalog",
        "BrushCatalog",
        "SymbolCatalog",
    )

    # Управляют доступом к UserDataPaths

    class _MappingReprStr(ABC):

        @abstractmethod
        def items(self) -> List[Tuple[str, Path]]: ...

        def __repr__(self) -> str:
            return repr(self.items())

        def __str__(self) -> str:
            return str(self.items())

    class _CustomMutableMapping(MutableMapping, _MappingReprStr):

        def __init__(self, dict_arg: dict):
            self._dict_arg = dict_arg

        @staticmethod
        def _reset_to_default() -> None:
            dict_ = dict(DefaultSettings.UserDataPaths.items())
            list_of_lists = [[k, str(v)] for k, v in dict_.items()]
            _ShadowSettingsManager.set_value("UserDataPaths", list_of_lists)

        @staticmethod
        def _ensure_existing_dir(v: Union[Path, str]) -> None:

            def check_is_dir(v_arg: Path):
                if not v_arg.is_dir():
                    raise TypeError("Needs path to existing dir")

            if isinstance(v, Path):
                check_is_dir(v)
            elif isinstance(v, str):
                v = Path(v)
                check_is_dir(v)
            else:
                raise TypeError("Needs Path or str value")

        def _update_setting(self) -> None:
            list_of_lists = [[k, str(v)] for k, v in self._dict_arg.items()]
            _ShadowSettingsManager.set_value("UserDataPaths", list_of_lists)

        def __setitem__(self, __k, __v) -> None:
            self._ensure_existing_dir(__v)
            self._dict_arg[__k] = __v
            self._update_setting()

        def __delitem__(self, __k) -> None:
            del self._dict_arg[__k]
            self._update_setting()

        def __getitem__(self, __k) -> Path:
            return self._dict_arg[__k]

        def __len__(self) -> int:
            return len(self._dict_arg)

        def __iter__(self) -> Iterator:
            return iter(self._dict_arg)

        def keys(self) -> List[str]:
            return list(super().keys())

        def values(self) -> List[Path]:
            return list(super().values())

        def items(self) -> List[Tuple[str, Path]]:
            return list(super().items())

    class _CustomMapping(Mapping, _MappingReprStr):

        def __init__(self, dict_arg: dict):
            self._dict_arg = dict_arg

        def __getitem__(self, __k) -> Path:
            return self._dict_arg[__k]

        def __len__(self) -> int:
            return len(self._dict_arg)

        def __iter__(self) -> Iterator:
            return iter(self._dict_arg)

        def keys(self) -> List[str]:
            return list(super().keys())

        def values(self) -> List[Path]:
            return list(super().values())

        def items(self) -> List[Tuple[str, Path]]:
            return list(super().items())

    # ----------------------------------

    @classmethod
    def _shadow_value_to_value(cls, key, shadow_value: ShadowValue, default) -> Value:
        """Преобразует значение из внутреннего формата."""
        value = shadow_value

        if key == "UserDataPaths" and isinstance(value, list):
            value = {k: Path(v) for k, v in value}
            if default:
                value = cls._CustomMapping(value)
            else:
                value = cls._CustomMutableMapping(value)

        elif key in cls._PATH_KEYS:
            value = Path(value)

        return value

    @classmethod
    def _value_to_shadow_value(cls, key: str, value: Value) -> ShadowValue:
        """Преобразует значение во внутренний формат."""
        if key in cls._PATH_KEYS:
            shadow_value = str(value)
        else:
            shadow_value = value

        return shadow_value

    @staticmethod
    def _to_shadow_key(key: str) -> DefaultSettingsList.Key:
        """Преобразует ключ настроек во внутренний формат."""
        return getattr(DefaultSettingsList, key)

    @classmethod
    def get_value(cls, key: str, /, is_default: bool) -> Value:
        if is_default:
            func = ShadowSettings.default_value
        else:
            func = ShadowSettings.value
        shadow_value = func(cls._to_shadow_key(key))
        value = cls._shadow_value_to_value(key, shadow_value, is_default)
        return value

    @classmethod
    def set_value(cls, key: str, value: Value) -> None:

        # Автоматическая проверка простых типов
        passed_type = type(value)
        expected_type = type(cls.get_value(key, is_default=True))
        if expected_type in (bool, int, float, str):
            if passed_type != expected_type:
                raise TypeError(f"Key type mismatch; expected {expected_type} but got {passed_type} instead.")

        shadow_value = cls._value_to_shadow_value(key, value)
        ShadowSettings.set_value(cls._to_shadow_key(key), shadow_value)


class _Settings(_MappingMetaDocumentation):

    class _KeysDesc:

        def __init__(self) -> None:
            self._keys = None

        def __get__(self, instance, owner) -> List:
            if self._keys is None:
                self._keys = [k for k, v in _Settings.__dict__.items() if isinstance(v, _Settings._Desc)]
            return self._keys

    _desc_keys = _KeysDesc()

    class _Desc:
        """Дескриптор управляет доступом к атрибутам, для классов DefaultSettings и
        CurrentSettings."""

        def __set_name__(self, owner, name: str) -> None:
            self._name = name

        def __get__(self, obj, objtype=None) -> Any:
            """Can't compare objtype, because of infinite recursion."""
            if type(objtype) is _DefaultSettingsMappingMetaExtended:
                value = _ShadowSettingsManager.get_value(self._name, is_default=True)
            elif type(objtype) is _CurrentSettingsMutableMappingMeta:
                value = _ShadowSettingsManager.get_value(self._name, is_default=False)
            else:
                raise RuntimeError
            return value

    class _LangDesc(_Desc):

        def __get__(self, obj, objtype=None) -> "AxiomaLanguage":
            if type(objtype) is _DefaultSettingsMappingMetaExtended:
                result_str = "ru"
            elif type(objtype) is _CurrentSettingsMutableMappingMeta:
                result_str = _shadow_manager.core.translationLanguage()
            else:
                raise RuntimeError("_LangDesc: unknown type.")

            result = AxiomaLanguage[result_str]
            return result

    class _UserDataDesc(_Desc):

        def __get__(self, obj, objtype=None) -> Any:
            """Can't compare objtype, because of infinite recursion."""
            user_data_folder = Path(gui_instance._shadow.installedPluginsPath())
            user_plugins_folder = user_data_folder / "installed_modules"
            if self._name == "UserDataFolder":
                return user_data_folder
            elif self._name == "UserPluginsFolder":
                return user_plugins_folder
            elif self._name == "UserPluginsSettingsFolder":
                return user_plugins_folder / "settings"
            elif self._name == "UserPluginsDataFolder":
                return user_plugins_folder / "data"
            elif self._name == "UserPluginsInstallationFolder":
                return user_plugins_folder / "modules"
            elif self._name == "UserPluginsDependenciesFolder":
                return user_plugins_folder / "dependencies"
            else:
                raise RuntimeError("_UserDataDesc: unknown attribute name.")

    Language: "AxiomaLanguage" = _LangDesc()
    """Язык Аксиомы."""
    SilentCloseWidget: bool = _Desc()
    """Подтверждать закрытие несохраненных данных."""
    SnapSensitiveRadius: int = _Desc()
    """Привязка узлов - размер"""
    EditNodeColor: QColor = _Desc()
    """Узлы при редактировании - цвет"""
    EditNodeSize: int = _Desc()
    """Узлы при редактировании - размер"""
    NearlyGeometriesTopology: bool = _Desc()
    """Перемещать узлы соседних объектов при редактировании."""
    NodesUpdateMode: bool = _Desc()
    """Объединять историю изменения узлов в режиме Форма."""
    ShowDrawingToolTip: bool = _Desc()
    """Показывать данные при рисовании."""
    CreateTabAfterOpen: bool = _Desc()
    """Создавать TAB при открытии."""
    RenameDataObjectFromTab: bool = _Desc()
    """Переименовывать открытый объект по имени TAB файла."""
    LastSavePath: Path = _Desc()
    """Последний пусть сохранения."""
    UseLastSelectedFilter: bool = _Desc()
    """Запоминать последний фильтр в диалоге открытия файлов."""
    SelectByInformationTool: bool = _Desc()
    """Инструмент «Информация» выбирает объект."""
    SaveAsToOriginalFileFolder: bool = _Desc()
    """Сохранять копию в каталог с исходным файлом."""
    LastNameFilter: str = _Desc()
    """Последний использованный фильтр файлов."""
    SensitiveMouse: int = _Desc()
    """Чувствительность мыши в пикселях."""
    ShowSplashScreen: bool = _Desc()
    """Отображать экран загрузки."""
    RulerColorLine: QColor = _Desc()
    """Линейка - цвет линии"""
    UseAntialiasing: bool = _Desc()
    """Использовать сглаживание при отрисовке."""
    ShowDegreeTypeNumeric: bool = _Desc()
    """Отображать градусы в формате Десятичное значение."""
    DrawCoordSysBounds: bool = _Desc()
    """Отображать границы мира."""
    PreserveScaleMap: bool = _Desc()
    """Сохранять масштаб при изменении размеров окна."""
    ShowMapScaleBar: bool = _Desc()
    """Показывать масштабную линейку."""
    ShowScrollOnMapView: bool = _Desc()
    """Показывать полосы прокрутки."""
    LoadLastWorkspace: bool = _Desc()
    """Загружать при старте последнее рабочее пространство."""
    ShowMeshLayout: bool = _Desc()
    """Отображать сетку привязки."""
    MeshSizeLayout: float = _Desc()
    """Размер ячейки."""
    SnapToMeshLayout: bool = _Desc()
    """Привязывать элементы отчета к сетке."""
    ShowMeshLegend: bool = _Desc()
    """Отображать сетку привязки."""
    MeshSizeLegend: float = _Desc()
    """Размер ячейки."""
    SnapToMeshLegend: bool = _Desc()
    """Привязывать к сетке."""
    LastOpenPath: Path = _Desc()
    """Последний каталог откуда открывались данные."""
    LastPathWorkspace: Path = _Desc()
    """Последний каталог к рабочему набору."""
    DefaultPathCache: Path = _Desc()
    """Каталог с кэшированными данными."""
    UserDataPaths: Dict[str, Path] = _Desc()
    """Список пользовательских каталогов с названиями."""
    EnableSmartTabs: bool = _Desc()
    """Умное переключение вкладок."""
    DistancePrecision: int = _Desc()
    """Точность по умолчанию для расстояний и площадей."""
    PenCatalog: Path = _Desc()
    """Каталог со стилями линий."""
    BrushCatalog: Path = _Desc()
    """Каталог со стилями заливок."""
    SymbolCatalog: Path = _Desc()
    """Каталог со стилями точек."""
    UseNativeFileDialog: bool = _Desc()
    """Пользоваться системными файловыми диалогами."""
    PhysicalScreenSize: float = _Desc()
    """Размер экрана в дюймах."""
    UserDataFolder: Path = _UserDataDesc()
    """Папка с пользовательскими данными Аксиомы."""
    UserPluginsFolder: Path = _UserDataDesc()
    """Корневая папка установленных плагинов Аксиомы."""
    UserPluginsSettingsFolder: Path = _UserDataDesc()
    """Папка с настройками установленных плагинов Аксиомы."""
    UserPluginsDataFolder: Path = _UserDataDesc()
    """Папка с данными установленных плагинов Аксиомы."""
    UserPluginsInstallationFolder: Path = _UserDataDesc()
    """Папка с установленными плагинами Аксиомы."""
    UserPluginsDependenciesFolder: Path = _UserDataDesc()
    """Папка с зависимостями установленных плагинов Аксиомы."""

    @classmethod
    def items(cls) -> List[Tuple[str, Any]]:
        """
        Возвращает список кортежей ключ-значение, где ключи - это атрибуты класса, a значения - это значения настроек.
        """
        return super().items()

    @classmethod
    def keys(cls) -> List[str]:
        """Возвращает список ключей, где ключи это атрибуты класса."""
        return super().keys()

    @classmethod
    def values(cls) -> List[Any]:
        """Возвращает список значений, где значения это значения настроек."""
        return super().values()

    @classmethod
    def get(cls, key: str, default_value: Optional[Any] = None):
        """Возвращает значение по ключу."""
        return super().get(key, default_value)


class _DefaultSettingsMappingMetaExtended(_MappingMetaExtended):

    def __iter__(cls) -> Iterator:
        return iter(_Settings._desc_keys)

    def __len__(cls) -> int:
        return len(_Settings._desc_keys)

    def __getitem__(cls, key: str) -> Any:
        try:
            result = getattr(DefaultSettings, key)
        except AttributeError:
            raise KeyError(key)
        else:
            return result


class _CurrentSettingsMutableMappingMeta(_DefaultSettingsMappingMetaExtended):

    def __getitem__(cls, key: str) -> Any:
        try:
            result = getattr(CurrentSettings, key)
        except AttributeError:
            raise KeyError(key)
        else:
            return result

    @staticmethod
    def _set(key: str, value: Any) -> None:
        v = _Settings.__dict__[key]
        if type(v) is not _Settings._Desc or key in ("UserDataPaths",):
            raise TypeError(f"Can't set attribute '{key}'.")
        _ShadowSettingsManager.set_value(key, value)

    def __setattr__(cls, key: str, value: Any, *args) -> None:
        cls._set(key, value)

    def __setitem__(cls, key: str, value: Any) -> None:
        cls._set(key, value)


class AxiomaLanguage(str, Enum):
    """Язык Аксиомы."""

    ru = "ru"
    """Русский язык."""
    en = "en"
    """Английский язык."""


class DefaultSettings(_Settings, metaclass=_DefaultSettingsMappingMetaExtended):
    """
    Настройки Аксиомы по умолчанию.

    Класс является статическим словарем, доступным только для чтения (:class:`collections.abc.Mapping`).
    Методы и атрибуты класса такие же как у класса :class:`axipy.CurrentSettings`, но доступны только для чтения.
    """

    Language: "AxiomaLanguage"
    """Язык Аксиомы."""
    SilentCloseWidget: bool
    """Подтверждать закрытие несохраненных данных."""
    SnapSensitiveRadius: int
    """Привязка узлов - размер"""
    EditNodeColor: QColor
    """Узлы при редактировании - цвет"""
    EditNodeSize: int
    """Узлы при редактировании - размер"""
    NearlyGeometriesTopology: bool
    """Перемещать узлы соседних объектов при редактировании."""
    NodesUpdateMode: bool
    """Объединять историю изменения узлов в режиме Форма."""
    ShowDrawingToolTip: bool
    """Показывать данные при рисовании."""
    CreateTabAfterOpen: bool
    """Создавать TAB при открытии."""
    RenameDataObjectFromTab: bool
    """Переименовывать открытый объект по имени TAB файла."""
    LastSavePath: Path
    """Последний пусть сохранения."""
    UseLastSelectedFilter: bool
    """Запоминать последний фильтр в диалоге открытия файлов."""
    SelectByInformationTool: bool
    """Инструмент «Информация» выбирает объект."""
    SaveAsToOriginalFileFolder: bool
    """Сохранять копию в каталог с исходным файлом."""
    LastNameFilter: str
    """Последний использованный фильтр файлов."""
    SensitiveMouse: int
    """Чувствительность мыши в пикселях."""
    ShowSplashScreen: bool
    """Отображать экран загрузки."""
    RulerColorLine: QColor
    """Линейка - цвет линии"""
    UseAntialiasing: bool
    """Использовать сглаживание при отрисовке."""
    ShowDegreeTypeNumeric: bool
    """Отображать градусы в формате Десятичное значение."""
    DrawCoordSysBounds: bool
    """Отображать границы мира."""
    PreserveScaleMap: bool
    """Сохранять масштаб при изменении размеров окна."""
    ShowMapScaleBar: bool
    """Показывать масштабную линейку."""
    ShowScrollOnMapView: bool
    """Показывать полосы прокрутки."""
    LoadLastWorkspace: bool
    """Загружать при старте последнее рабочее пространство."""
    ShowMeshLayout: bool
    """Отображать сетку привязки."""
    MeshSizeLayout: float
    """Размер ячейки."""
    SnapToMeshLayout: bool
    """Привязывать элементы отчета к сетке."""
    ShowMeshLegend: bool
    """Отображать сетку привязки."""
    MeshSizeLegend: float
    """Размер ячейки."""
    SnapToMeshLegend: bool
    """Привязывать к сетке."""
    LastOpenPath: Path
    """Последний каталог откуда открывались данные."""
    LastPathWorkspace: Path
    """Последний каталог к рабочему набору."""
    DefaultPathCache: Path
    """Каталог с кэшированными данными."""
    UserDataPaths: Dict[str, Path]
    """Список пользовательских каталогов с названиями."""
    EnableSmartTabs: bool
    """Умное переключение вкладок."""
    DistancePrecision: int
    """Точность по умолчанию для расстояний и площадей."""
    PenCatalog: Path
    """Каталог со стилями линий."""
    BrushCatalog: Path
    """Каталог со стилями заливок."""
    SymbolCatalog: Path
    """Каталог со стилями точек."""
    UseNativeFileDialog: bool
    """Пользоваться системными файловыми диалогами."""
    UserDataFolder: Path
    """Папка с пользовательскими данными Аксиомы."""
    UserPluginsFolder: Path
    """Корневая папка установленных плагинов Аксиомы."""
    UserPluginsSettingsFolder: Path
    """Папка с настройками установленных плагинов Аксиомы."""
    UserPluginsDataFolder: Path
    """Папка с данными установленных плагинов Аксиомы."""
    UserPluginsInstallationFolder: Path
    """Папка с установленными плагинами Аксиомы."""
    UserPluginsDependenciesFolder: Path
    """Папка с зависимостями установленных плагинов Аксиомы."""


class CurrentSettings(_Settings, metaclass=_CurrentSettingsMutableMappingMeta):
    """
    Текущие настройки Аксиомы. Класс является статическим словарем
    (:class:`collections.abc.MutableMapping`). Поддерживает обращение по индексу.

    Note:
        Для получения настроек по умолчанию, используется класс :class:`axipy.DefaultSettings`.

    .. literalinclude:: /../../tests/doc_examples/test_example_settings.py
        :caption: Пример работы с настройками.
        :pyobject: test_example_settings
        :start-after: # start
        :end-before: # finish
        :dedent: 4
    """

    Language: "AxiomaLanguage"
    """Язык Аксиомы."""
    SilentCloseWidget: bool
    """Подтверждать закрытие несохраненных данных."""
    SnapSensitiveRadius: int
    """Привязка узлов - размер"""
    EditNodeColor: QColor
    """Узлы при редактировании - цвет"""
    EditNodeSize: int
    """Узлы при редактировании - размер"""
    NearlyGeometriesTopology: bool
    """Перемещать узлы соседних объектов при редактировании."""
    NodesUpdateMode: bool
    """Объединять историю изменения узлов в режиме Форма."""
    ShowDrawingToolTip: bool
    """Показывать данные при рисовании."""
    CreateTabAfterOpen: bool
    """Создавать TAB при открытии."""
    RenameDataObjectFromTab: bool
    """Переименовывать открытый объект по имени TAB файла."""
    LastSavePath: Path
    """Последний пусть сохранения."""
    UseLastSelectedFilter: bool
    """Запоминать последний фильтр в диалоге открытия файлов."""
    SelectByInformationTool: bool
    """Инструмент «Информация» выбирает объект."""
    SaveAsToOriginalFileFolder: bool
    """Сохранять копию в каталог с исходным файлом."""
    LastNameFilter: str
    """Последний использованный фильтр файлов."""
    SensitiveMouse: int
    """Чувствительность мыши в пикселях."""
    ShowSplashScreen: bool
    """Отображать экран загрузки."""
    RulerColorLine: QColor
    """Линейка - цвет линии"""
    UseAntialiasing: bool
    """Использовать сглаживание при отрисовке."""
    ShowDegreeTypeNumeric: bool
    """Отображать градусы в формате Десятичное значение."""
    DrawCoordSysBounds: bool
    """Отображать границы мира."""
    PreserveScaleMap: bool
    """Сохранять масштаб при изменении размеров окна."""
    ShowMapScaleBar: bool
    """Показывать масштабную линейку."""
    ShowScrollOnMapView: bool
    """Показывать полосы прокрутки."""
    LoadLastWorkspace: bool
    """Загружать при старте последнее рабочее пространство."""
    ShowMeshLayout: bool
    """Отображать сетку привязки."""
    MeshSizeLayout: float
    """Размер ячейки."""
    SnapToMeshLayout: bool
    """Привязывать элементы отчета к сетке."""
    ShowMeshLegend: bool
    """Отображать сетку привязки."""
    MeshSizeLegend: float
    """Размер ячейки."""
    SnapToMeshLegend: bool
    """Привязывать к сетке."""
    LastOpenPath: Path
    """Последний каталог откуда открывались данные."""
    LastPathWorkspace: Path
    """Последний каталог к рабочему набору."""
    DefaultPathCache: Path
    """Каталог с кэшированными данными."""
    UserDataPaths: Dict[str, Path]
    """Список пользовательских каталогов с названиями."""
    EnableSmartTabs: bool
    """Умное переключение вкладок."""
    DistancePrecision: int
    """Точность по умолчанию для расстояний и площадей."""
    PenCatalog: Path
    """Каталог со стилями линий."""
    BrushCatalog: Path
    """Каталог со стилями заливок."""
    SymbolCatalog: Path
    """Каталог со стилями точек."""
    UseNativeFileDialog: bool
    """Пользоваться системными файловыми диалогами."""
    UserDataFolder: Path
    """Папка с пользовательскими данными Аксиомы."""
    UserPluginsFolder: Path
    """Корневая папка установленных плагинов Аксиомы."""
    UserPluginsSettingsFolder: Path
    """Папка с настройками установленных плагинов Аксиомы."""
    UserPluginsDataFolder: Path
    """Папка с данными установленных плагинов Аксиомы."""
    UserPluginsInstallationFolder: Path
    """Папка с установленными плагинами Аксиомы."""
    UserPluginsDependenciesFolder: Path
    """Папка с зависимостями установленных плагинов Аксиомы."""

    @staticmethod
    def reset() -> None:
        """Сброс значений всех текущих настроек на значения по умолчанию."""
        for k, v in DefaultSettings.items():
            if k == "UserDataPaths":
                cast(
                    _ShadowSettingsManager._CustomMutableMapping,
                    CurrentSettings.UserDataPaths,
                )._reset_to_default()
                continue
            try:
                CurrentSettings[k] = v
            except TypeError:
                continue
