# import logging
from dataclasses import dataclass
from pathlib import Path
from typing import TYPE_CHECKING

from PySide2.QtGui import QTransform
from axipy import (provider_manager,
                   Table, AxipyProgressHandler, Schema, Source, Destination,
                   Feature, CoordSystem)

if TYPE_CHECKING:
    from .register_vector_dialog import RegisterVectorDialog


class Exporter:
    @dataclass
    class Result:
        successfully_exported_elements: int
        all_elements: int

    def __init__(
            self,
            *,
            inp_table: Table,
            target_cs: CoordSystem,
            q_transform: QTransform,
            out_file_path: Path,
            widget_dialog: 'RegisterVectorDialog',
    ) -> None:
        self.counter = 0
        self._inp_table = inp_table
        self._target_cs = target_cs
        self._q_transform = q_transform
        self._out_file_path = out_file_path
        self._widget_dialog: RegisterVectorDialog = widget_dialog

    @property
    def inp_table(self) -> Table:
        return self._inp_table

    @property
    def q_transform(self) -> QTransform:
        return self._q_transform

    @property
    def out_file_path(self) -> Path:
        return self._out_file_path

    @property
    def widget_dialog(self) -> 'RegisterVectorDialog':
        return self._widget_dialog

    def export(
            self,
            ph: AxipyProgressHandler,
    ) -> Result:
        table = self.inp_table
        n = table.count()
        schema = table.schema
        schema.coordsystem = self._target_cs
        destination = self.get_destination_tab_from_path(self.out_file_path, schema)
        features = table.items()
        features = map(self.process_single_feature, features)
        self.counter = 0

        def func_callback(_f: Feature, _i: int):
            self.widget_dialog.signal_pbar_features_add_value.emit()
            if ph.is_canceled():
                return False

            self.counter += 1

        # logging.debug(f"{destination=}")
        destination.export(features, func_callback=func_callback)
        return self.Result(self.counter, n)

    def process_single_feature(self, f: Feature) -> Feature:
        g = f.geometry
        if g is None:
            return f
        else:
            g = g.affine_transform(self.q_transform)
            g.coordsystem = self._target_cs
            f.geometry = g
            return f

    @staticmethod
    def get_source_tab_from_path(path: Path, hidden: bool = True) -> Source:
        source = provider_manager.tab.get_source(str(path))
        source["hidden"] = hidden
        return source

    @staticmethod
    def get_destination_tab_from_path(path: Path, schema: Schema = None) -> Destination:
        if schema is None:
            schema = Schema()
        destination = provider_manager.tab.get_destination(str(path), schema)
        return destination
