from typing import Optional

import axipy
from axipy import Plugin
from axipy.da import (
    CollectionStyle,
    Feature,
    GeometryType,
    PointPictureStyle,
    Style,
    Table,
)
from axipy.gui import view_manager

from ..data_types import NamedSearchPatterns, SearchGroupPatterns, SearchPatterns
from ..widgets.select_by_style_widget import SelectByStyleWidget
from ..widgets.style_widget_interface import SelectByStyleWidgetInterface
from .strategy_base import SelectStrategy


class RasterPointStrategy(SelectStrategy):
    """
    Стратегия выделения для точек со стилем в виде растрового изображения
    """

    def __init__(self, f: Feature, table: Table) -> None:
        super().__init__(f, table)
        self.__color = axipy.tr("Цвет")
        self.__size = axipy.tr("Размер")
        self.__pattern = axipy.tr("Паттерн")
        self.__geometries = [GeometryType.Point, GeometryType.MultiPoint, GeometryType.GeometryCollection]

    def fill_widget(self, widget: "SelectByStyleWidget"):
        widget.search_geometries = {gtype: True for gtype in self.__geometries}
        widget.search_patterns = NamedSearchPatterns(self.search_patterns(), self.string_pattern_group_key())

    def search_patterns(self) -> SearchPatterns:
        return {self.__color: True, self.__size: True, self.__pattern: True}

    def identify(self, style: Style) -> bool:
        return isinstance(style, PointPictureStyle)

    def string_pattern_group_key(self) -> str:
        return "raster_point_key"

    def settings_widget(self, iface: Plugin) -> SelectByStyleWidgetInterface:
        widget = SelectByStyleWidget(view_manager.global_parent, iface)
        widget.put_content_on_top(True)
        self.fill_widget(widget)
        return widget

    def extract_style_from_collection(self, collection_style: CollectionStyle) -> Optional[Style]:
        pstyle = collection_style.point
        if pstyle:
            return pstyle.clone()
        return pstyle

    def is_supported_style(
        self, target: PointPictureStyle, source: PointPictureStyle, group_patterns: SearchGroupPatterns
    ) -> bool:
        assert isinstance(target, PointPictureStyle)
        assert isinstance(source, PointPictureStyle)
        patterns = group_patterns[self.string_pattern_group_key()]
        if source.filename != target.filename:
            return False
        if patterns[self.__color] and source.color != target.color:
            return False
        if patterns[self.__size] and source.size != target.size:
            return False
        pattern_check = source.actual_size != target.actual_size or source.apply_color != source.apply_color
        if patterns[self.__pattern] and pattern_check:
            return False
        return True
