from typing import Any, Generator, Iterable, List, Tuple, cast

import axipy

from .utils import (
    Direction,
    _flat_geom_generator,
    _init_feature_geom_generator,
    get_direction,
)

__all__ = [
    "Worker",
]


# noinspection PyMethodMayBeStatic
class Worker:

    def __init__(self, outer_direction_calculated: Direction, inner_direction_calculated: Direction) -> None:
        self.outer_direction_calculated: Direction = outer_direction_calculated
        self.inner_direction_calculated: Direction = inner_direction_calculated

        self.updated_features_counter: int = 0

    def direction_to_line_direction(self, d: Direction) -> axipy.LineDirection:
        if d == Direction.CLOCKWISE:
            return axipy.LineDirection.Clockwise
        elif d == Direction.COUNTERCLOCKWISE:
            return axipy.LineDirection.CounterClockwise
        raise ValueError(f"{d}")

    def updated_features_generator(
        self,
        items: Iterable[axipy.Feature],
        count: int,
        outer_contours_direction: Direction,
        holes_exist: bool,
        inner_contours_direction: Direction,
        task: axipy.DialogTask,
    ) -> Generator[axipy.Feature, Any, None]:
        skip_outer = self.outer_direction_calculated == outer_contours_direction
        skip_inner = self.inner_direction_calculated == inner_contours_direction
        if not holes_exist:
            skip_inner = True
        skip_both = skip_outer and skip_inner
        if skip_both:
            return None

        self.updated_features_counter = 0
        for f, g in _init_feature_geom_generator(items, count, task):
            if isinstance(g, axipy.Polygon):
                feature_needs_update = self.modify_polygon(
                    g, outer_contours_direction, inner_contours_direction, skip_outer, skip_inner
                )
                if feature_needs_update:
                    f.geometry = g
                    self.updated_features_counter += 1
                    yield f
            elif isinstance(g, axipy.LineString):
                feature_needs_update = self.modify_closed_line_string(
                    g,
                    outer_contours_direction,
                    inner_contours_direction,
                    skip_outer,
                    skip_inner,
                )
                if feature_needs_update:
                    f.geometry = g
                    self.updated_features_counter += 1
                    yield f
            elif isinstance(g, axipy.GeometryCollection):
                g_col_needs_update: bool = False
                for elem in g:
                    if isinstance(elem, axipy.Polygon):
                        polygon_needs_update = self.modify_polygon(
                            elem, outer_contours_direction, inner_contours_direction, skip_outer, skip_inner
                        )
                        if not g_col_needs_update and polygon_needs_update:
                            g_col_needs_update = True
                    elif isinstance(elem, axipy.LineString):
                        polygon_needs_update = self.modify_closed_line_string(
                            elem,
                            outer_contours_direction,
                            inner_contours_direction,
                            skip_outer,
                            skip_inner,
                        )
                        if not g_col_needs_update and polygon_needs_update:
                            g_col_needs_update = True
                if g_col_needs_update:
                    f.geometry = g
                    self.updated_features_counter += 1
                    yield f

    # inplace
    def modify_polygon(
        self,
        g: axipy.Polygon,
        outer_contours_direction: Direction,
        inner_contours_direction: Direction,
        skip_outer: bool,
        skip_inner: bool,
    ) -> bool:
        feature_needs_update: bool = False

        # outer
        if not skip_outer:
            points = cast(axipy.ListPoint, g.points)

            direction = get_direction(points)
            if direction != outer_contours_direction:
                self.reverse_list_points(points)
                feature_needs_update = True

        # inner (holes)
        if skip_inner:
            return feature_needs_update

        holes_list = cast(List[axipy.ListPoint], g.holes)
        for points in holes_list:
            direction = get_direction(points)
            if direction != inner_contours_direction:
                self.reverse_list_points(points)
                feature_needs_update = True

        return feature_needs_update

    # inplace
    def modify_closed_line_string(
        self,
        g: axipy.LineString,
        outer_contours_direction: Direction,
        _inner_contours_direction: Direction,
        skip_outer: bool,
        _skip_inner: bool,
    ) -> bool:
        if not g._shadow.is_closed():
            return False

        feature_needs_update: bool = False

        # outer
        if not skip_outer:
            points = cast(axipy.ListPoint, g.points)

            direction = get_direction(points)
            if direction != outer_contours_direction:
                self.reverse_list_points(points)
                feature_needs_update = True

        return feature_needs_update

    # inplace
    def reverse_list_points(self, list_points: axipy.ListPoint) -> None:
        size: int = len(list_points)
        i: int = 0
        while i < size // 2:
            swap_index: int = size - i - 1
            list_points[i], list_points[swap_index] = list_points[swap_index], list_points[i]
            i += 1

    def change_direction(
        self,
        task: axipy.DialogTask,
        outer_contours_direction: Direction,
        holes_exist: bool,
        inner_contours_direction: Direction,
    ) -> Tuple[int, int]:
        task.title = "Изменение направления контуров"
        task.message = task.title

        table = axipy.data_manager.selection
        count = table.count()
        task.max = count

        table.update(
            self.updated_features_generator(
                table.items(),
                table.count(),
                outer_contours_direction,
                holes_exist,
                inner_contours_direction,
                task,
            )
        )

        if task.is_canceled and (task.value > 0):
            if table.can_undo:
                table.undo()
                self.updated_features_counter = 0

        return count, self.updated_features_counter


# noinspection PyMethodMayBeStatic
class ContoursDirectionCalculator(axipy.DialogTask):

    def __init__(self) -> None:
        super().__init__(self.calculate_contours_direction, cancelable=True)

        self.clockwise_exists: bool = False
        self.counterclockwise_exists: bool = False

        self.holes_exist: bool = False
        self.holes_clockwise_exists: bool = False
        self.holes_counterclockwise_exists: bool = False

        # self.clockwise_count: int = 0
        # self.counterclockwise_count: int = 0
        # self.mixed_count: int = 0
        # self.holes_clockwise_count: int = 0
        # self.holes_counterclockwise_count: int = 0
        # self.holes_mixed_count: int = 0
        # # obj count
        # self.polygons_count: int = 0
        # self.polygons_count_det_zero: int = 0
        # self.holes_count: int = 0
        # self.holes_count_det_zero: int = 0
        # self.closed_line_string_count: int = 0
        # self.closed_line_string_count_det_zero: int = 0

    # def print_count(self) -> None:
    #     print(f"{self.polygons_count=}")
    #     print(f"{self.polygons_count_det_zero=}")
    #     print(f"{self.holes_count=}")
    #     print(f"{self.holes_count_det_zero=}")
    #     print(f"{self.closed_line_string_count=}")
    #     print(f"{self.closed_line_string_count_det_zero=}")
    #     print()
    #     print(f"{self.clockwise_count=}")
    #     print(f"{self.counterclockwise_count=}")
    #     print(f"{self.mixed_count=}")
    #     print(f"{self.holes_exist=}")
    #     print(f"{self.holes_clockwise_count=}")
    #     print(f"{self.holes_counterclockwise_count=}")
    #     print(f"{self.holes_mixed_count=}")
    #     print()

    def find_holes(self, single_g: axipy.Geometry) -> bool:
        if isinstance(single_g, axipy.Polygon):
            return len(single_g.holes) > 0
        return False

    def process_polygon(self, p: axipy.Polygon) -> None:
        points = cast(axipy.ListPoint, p.points)

        direction = get_direction(points)
        if direction == Direction.CLOCKWISE:
            # self.clockwise_count += 1
            self.clockwise_exists = True
        elif direction == Direction.COUNTERCLOCKWISE:
            # self.counterclockwise_count += 1
            self.counterclockwise_exists = True
        # else:
        #     self.polygons_count_det_zero += 1

        if not all(
            (
                self.holes_clockwise_exists,
                self.holes_counterclockwise_exists,
            )
        ):
            holes_list = cast(List[axipy.ListPoint], p.holes)
            for hole in holes_list:
                # self.holes_count += 1
                if not self.holes_exist:
                    self.holes_exist = True
                self.process_hole(hole)
                if all(
                    (
                        self.holes_clockwise_exists,
                        self.holes_counterclockwise_exists,
                    )
                ):
                    break

    def process_hole(self, h: axipy.ListPoint) -> None:
        direction = get_direction(h)
        if direction == Direction.CLOCKWISE:
            # self.holes_clockwise_count += 1
            self.holes_clockwise_exists = True
        elif direction == Direction.COUNTERCLOCKWISE:
            # self.holes_counterclockwise_count += 1
            self.holes_counterclockwise_exists = True
        # else:
        #     self.holes_count_det_zero += 1

    def process_closed_line_string(self, ls: axipy.LineString) -> None:
        points = cast(axipy.ListPoint, ls.points)

        direction = get_direction(points)
        if direction == Direction.CLOCKWISE:
            # self.clockwise_count += 1
            self.clockwise_exists = True
        elif direction == Direction.COUNTERCLOCKWISE:
            # self.counterclockwise_count += 1
            self.counterclockwise_exists = True
        # else:
        #     self.closed_line_string_count_det_zero += 1

    def run_and_get(self, selection_table: axipy.SelectionTable) -> Tuple[Direction, bool, Direction]:
        return super().run_and_get(selection_table)

    def calculate_contours_direction(self, selection_table: axipy.SelectionTable) -> Tuple[Direction, bool, Direction]:
        self.title = "Вычисление направления контуров"
        self.message = self.title

        outer_contours_direction: Direction = Direction.MIXED
        inner_contours_direction: Direction = Direction.MIXED

        for g in _flat_geom_generator(selection_table.items(), selection_table.count(), self):
            if isinstance(g, axipy.Polygon):
                # self.polygons_count += 1
                self.process_polygon(g)
            elif isinstance(g, axipy.LineString) and g._shadow.is_closed():
                # self.closed_line_string_count += 1
                self.process_closed_line_string(g)

            if all(
                (
                    self.clockwise_exists,
                    self.counterclockwise_exists,
                    self.holes_clockwise_exists,
                    self.holes_counterclockwise_exists,
                )
            ):
                break

        if self.clockwise_exists and self.counterclockwise_exists:
            outer_contours_direction = Direction.MIXED
        elif self.counterclockwise_exists:
            outer_contours_direction = Direction.COUNTERCLOCKWISE
        elif self.clockwise_exists:
            outer_contours_direction = Direction.CLOCKWISE

        if self.holes_clockwise_exists and self.holes_counterclockwise_exists:
            inner_contours_direction = Direction.MIXED
        elif self.holes_counterclockwise_exists:
            inner_contours_direction = Direction.COUNTERCLOCKWISE
        elif self.holes_clockwise_exists:
            inner_contours_direction = Direction.CLOCKWISE

        return outer_contours_direction, self.holes_exist, inner_contours_direction
