from typing import TYPE_CHECKING

import axipy
from PySide2.QtCore import Qt
from PySide2.QtGui import QIcon
from .dialog import Dialog
from .observer import RedirectContoursObserver, _check_if_any_closed_contours_selected


if TYPE_CHECKING:
    from ...__init__ import CadTools

__all__ = [
    "RedirectContoursActionButton",
]


class RedirectContoursActionButton(axipy.ActionButton):

    def __init__(
        self,
        plugin: "CadTools",
    ) -> None:
        self.plugin: "CadTools" = plugin
        self.title: str = self.plugin.tr("Изменить направление контуров")
        self.icon: QIcon = QIcon.fromTheme("contour_redirect")
        self.tooltip: str = self.plugin.tr("Изменение направления контуров для выбранных объектов.")
        self.redirect_contours_observer: RedirectContoursObserver = RedirectContoursObserver()
        super().__init__(self.title, self.on_click, self.icon, self.redirect_contours_observer, self.tooltip)

    def on_click(self) -> None:
        dt = axipy.DialogTask(_check_if_any_closed_contours_selected)
        if dt.run_and_get():
            dialog_ = Dialog(self.plugin, axipy.view_manager.global_parent)
            dialog_.setAttribute(Qt.WA_DeleteOnClose, True)
            dialog_.setWindowIcon(self.icon)
            dialog_.open()
        else:
            axipy.show_message(
                title=self.plugin.tr("Предупреждение"),
                text=self.plugin.tr("Не выбрано ни одного полигона или замкнутой полилинии."),
                icon=axipy.DlgIcon.WARNING,
            )

    def remove(self) -> None:
        self.redirect_contours_observer.unload()
        super().remove()
