from axipy.da.data_object import Table
from axipy.cpp_core_dp import ShadowConverter
from .data_provider import DataProvider
from .source import Source
from typing import List
from enum import Enum


__all__ = [
    "DwgSource",
    "DwgDataProvider",
    "DwgFileVersion",
    "DwgFileFormat",
    "DwgPalette"
]


class DwgSource(Source):
    pass


class DwgFileFormat(int, Enum):
    """Формат файла AutoCAD. Используется при задании параметра в функции :meth:`DwgDataProvider.convert_file`"""

    Dwg   = 0
    """Файл DWG"""
    Dxf   = 1
    """Текстовый файл DXF"""
    Dxb   = 2
    """Бинарный файл DXF"""


class DwgFileVersion(int, Enum):
    """Версия файла DWG AutoCAD. Используется при задании параметра в функции :meth:`DwgDataProvider.convert_file`"""

    AutoCAD_R9 =  11
    """AutoCAD Release 9"""
    AutoCAD_R10 = 13
    """AutoCAD Release 10"""
    AutoCAD_R11_12 = 16
    """AutoCAD Release 11-12"""
    AutoCAD_R13 = 19
    """AutoCAD Release 13"""
    AutoCAD_R14  = 21
    """AutoCAD Release 14"""
    AutoCAD_2000 = 23
    """AutoCAD 2000-2002"""
    AutoCAD_2004 = 25
    """AutoCAD 2004-2006"""
    AutoCAD_2007 = 27
    """AutoCAD 2007-2009"""
    AutoCAD_2010 = 29
    """AutoCAD 2010-2012"""
    AutoCAD_2013 = 31
    """AutoCAD 2013-2016"""

class DwgPalette(int, Enum):
    """Палитра при работе с файлами AutoCAD. Используется при задании параметра в функции :meth:`DwgDataProvider.set_palette`"""

    Light = 1
    """Светлая тема"""
    Dark = 2
    """Темная тема"""


class DwgDataProvider(DataProvider):
    """Провайдер для источников формата AutoCAD.
    
    Note:
        Ссылку на провайдер можно получить через глобальную переменную :attr:`axipy.provider_manager.dwg`.
    """
    _identifier = 'DwgDgnFileProvider'

    def get_source(self, data: str, alias: str = None) -> Source:
        """Создает источник данных.

        Args:
          data: Имя файла или описание источника данных.
        """
        return DwgSource(
            Source._provider(self.id),
            Source._alias(alias),
            {
                'src': data,
            }
        )

    def open(self, data: str, alias: str = None) -> Table:
        """Открывает объект данных.

        Args:
          data: Имя файла или описание источника данных.
          alias: Псевдоним для открываемого объекта.
        """
        return self.get_source(data, alias).open()

    def get_destination(self):
        """
        Attention:
            Не поддерживается.

        Raises:
            NotImplementedError
        """
        raise NotImplementedError

    def create_open(self):
        """
        Attention:
            Не поддерживается.

        Raises:
            NotImplementedError
        """
        raise NotImplementedError

    def file_extensions(self) -> List[str]:
        return list(filter(lambda v: v in ['dwg', 'dxf'], super().file_extensions()))


    def convert_file(self, src_filepath: str, dest_filepath: str, out_version: DwgFileVersion = DwgFileVersion.AutoCAD_R13,
                         out_format: DwgFileFormat = DwgFileFormat.Dxf):
        """Производит конвертацию исходный файл текущего провайдера в другой формат этого же провайдера.

        Args:
            src_filepath: Путь к исходному файлу (имя файла).
            dest_filepath: Путь к выходному файлу (имя файла).
            out_version: Версия выходного файла
            out_format: Формат выходного файла


        .. code-block:: python

            input_file = 'filename_in.dwg'
            output_file = 'filename_out.dxf'
            provider_manager.dwg.convert_file(input_file, output_file, out_format = DwgFileFormat.Dxf, out_version = DwgFileVersion.AutoCAD_R13)

        """
        from os import path
        if not path.exists(src_filepath):
            raise FileNotFoundError(f"Файл '{src_filepath}' не существует")

        convert_data = {
            "openWith": self.id,
            "src": src_filepath,
            "dest": dest_filepath,
            "out_format": int(out_format),
            "out_version": int(out_version)
        }
        ShadowConverter.convertAcadFile(convert_data)

    def set_palette(self, palette: DwgPalette):
        """Устанавливает текущую палитру.

        Args:
            palette: Индекс палитры.
        """
        data = {
            "openWith": self.id,
            "num_palette": int(palette)
        }
        ShadowConverter.setAcadPalette(data)