from abc import abstractmethod
from collections.abc import KeysView, ItemsView, ValuesView, Mapping
from typing import Tuple, List, Any, Iterator


class _NoCallMeta(type):
    def __call__(cls, *args, **kwargs) -> None:
        raise TypeError("Class doesn't support instance creation")


class _NoSetAttrMeta(type):
    def __setattr__(cls, key: Any, value: Any, *args) -> None:
        if "override" in args:
            type.__setattr__(cls, key, value)
        else:
            raise TypeError("Class is read-only")


# collections abc meta


class _SizedMeta(type):

    @abstractmethod
    def __len__(cls) -> int:
        raise NotImplementedError


class _IterableMeta(type):

    @abstractmethod
    def __iter__(cls) -> Iterator:
        raise NotImplementedError


class _ContainerMeta(type):

    @abstractmethod
    def __contains__(cls, item: Any) -> bool:
        raise NotImplementedError


class _CollectionMeta(_SizedMeta, _IterableMeta, _ContainerMeta):
    ...


class _MappingMeta(_CollectionMeta):

    @abstractmethod
    def __getitem__(cls, key: Any) -> Any:
        raise NotImplementedError

    # Default
    def __contains__(cls, key):
        try:
            cls[key]
        except KeyError:
            return False
        else:
            return True

    def keys(cls):
        return list(KeysView(cls))

    def values(cls):
        return list(ValuesView(cls))

    def items(cls):
        return list(ItemsView(cls))

    def get(cls, key, default=None):
        try:
            return cls[key]
        except KeyError:
            return default

    def __eq__(cls, other):
        if not isinstance(other, _MappingMeta):
            return NotImplemented
        return dict(cls.items()) == dict(other.items())


class _MappingMetaReprStr(type):

    def __repr__(cls):
        return repr(cls.items())

    def __str__(cls):
        return str(cls.items())


class _MappingMetaExtended(_NoCallMeta, _NoSetAttrMeta, _MappingMetaReprStr, _MappingMeta):
    ...


class _MappingMetaDocumentation:
    """
    Нужен для документации sphinx и для IntelliSense в IDE.
    Методы абстрактные, чтобы наследник переопределял docstring и type-hint.
    При наследовании нужен только вызов super().
    """

    @classmethod
    def items(cls) -> List[Tuple[str, Any]]:
        """
        Возвращает список кортежей ключ-значение,
        где ключи это строки,
        а значения это объекты класса :class:`object`.
        """
        return type(cls).items(cls)

    @classmethod
    def keys(cls) -> List[str]:
        """
        Возвращает список ключей,
        где ключи это строки.
        """
        return type(cls).keys(cls)

    @classmethod
    def values(cls) -> List[Any]:
        """
        Возвращает список значений,
        где значения это объекты класса :class:`object`.
        """
        return type(cls).values(cls)

    @classmethod
    def get(cls, key: str, default_value: Any = None) -> Any:
        """
        Возвращает значение по ключу.
        Если ключа нет в словаре, возвращает значение по умолчанию.
        """
        return type(cls).get(cls, key, default_value)


class _MutableMappingMetaDocumentation(_MappingMetaDocumentation):
    __marker = object()

    @classmethod
    def pop(cls, key, default=__marker) -> Any:
        return type(cls).pop(cls, key, default)

    @classmethod
    def popitem(cls) -> Any:
        return type(cls).popitem(cls)

    @classmethod
    def clear(cls) -> None:
        type(cls).clear(cls)

    @classmethod
    def update(cls, iterable=(), /, **keywords) -> None:
        type(cls).update(cls, iterable, **keywords)

    @classmethod
    def setdefault(cls, key, default=None) -> Any:
        return type(cls).setdefault(cls, key, default)


class _MutableMappingMetaExtended(_MappingMetaExtended):

    @abstractmethod
    def __setitem__(cls, key, value) -> None:
        raise KeyError

    @abstractmethod
    def __delitem__(cls, key) -> None:
        raise KeyError

    __marker = object()

    def pop(cls, key, default=__marker) -> Any:
        try:
            value = cls[key]
        except KeyError:
            if default is cls.__marker:
                raise
            return default
        else:
            del cls[key]
            return value

    def popitem(cls):
        try:
            key = next(iter(cls))
        except StopIteration:
            raise KeyError from None
        value = cls[key]
        del cls[key]
        return key, value

    def clear(cls) -> None:
        try:
            while True:
                type(cls).popitem(cls)
        except KeyError:
            pass

    def update(cls, other=(), /, **kwds):
        if isinstance(other, Mapping):
            for key in other:
                cls[key] = other[key]
        elif hasattr(other, "keys"):
            for key in other.keys():
                cls[key] = other[key]
        else:
            for key, value in other:
                cls[key] = value
        for key, value in kwds.items():
            cls[key] = value

    def setdefault(cls, key, default=None):
        try:
            return cls[key]
        except KeyError:
            cls[key] = default
        return default
