"""Helper functions."""
import warnings
from functools import partial, wraps


class InitOnce:
    """Runs function once and then keeps returning the result."""

    def __init__(self, f):
        self.f = f
        self.result = None

    def __call__(self, *args, **kwargs):
        if self.result is None:
            self.result = self.f(*args, **kwargs)
        return self.result

    def __get__(self, instance, owner):
        return partial(self, instance)


def _deprecated_by(use_instead=None):
    def decorate(func):

        # Автоматическое исключение задекорированных методов sphinx-ом.
        doc = func.__doc__
        if isinstance(doc, str) and ":meta private:" not in doc:
            func.__doc__ += "\n:meta private:"

        @wraps(func)
        def wrapper(*args, **kwargs):
            if use_instead is None:
                msg = f"'{func.__name__}' is deprecated."
            else:
                msg = f"'{func.__name__}' is deprecated, use '{use_instead}' instead."
            warnings.warn(msg, DeprecationWarning, stacklevel=2)
            result = func(*args, **kwargs)
            return result

        return wrapper

    return decorate
