import os.path
from pathlib import Path
from typing import Union

from axipy.cpp_gui import ShadowWorkspace

from .gui_class import gui_instance

__all__ = ["Workspace"]


class Workspace:
    """
    Сохранение рабочего пространства в файл рабочего набора ``*.mws``.

    Рабочее пространство можно как сохранять в файл так и читать из него.
    При чтении из файла рабочего пространства посредством метода :meth:`load_file`
    все источники данных (таблицы) открываются и добавляются в каталог :class:`axipy.DataManager`,
    доступный через переменную :attr:`axipy.data_manager`.
    А окна с наполнением добавляются в менеджер окон :class:`axipy.ViewManager`,
    доступный через переменную :attr:`view_manager`.
    
    В случае записи текущего состояния в файл рабочего пространства последовательность обратная рассмотренной.
    Состояние каталога и менеджера окон записывается в рабочее пространство посредством метода :meth:`save_file`.

    Пример чтения данных в рамках отдельного приложения::

        from axipy import init_axioma, data_manager, view_manager, Workspace, open_file_dialog

        # инициализация ядра
        app = init_axioma()
        print('Before: tables({}), views({})'.format(len(data_manager), len(view_manager)))
        # Чтение рабочего набора
        file_name = open_file_dialog("Рабочий набор (*.mws)")
        Workspace().load_file(file_name)
        print('After: tables({}), views({})'.format(len(data_manager), len(view_manager)))
        # Если есть хотя бы одно окно с картой, показываем его
        if len(view_manager.mapviews):
            mapview = view_manager.mapviews[0]
            mapview.show()
        app.exec_()

        # Результат:
        # Before: tables(0), views(0)
        # After: tables(5), views(3)  # В зависимости от рабочего набора, числа могут отличаться


    Пример записи рабочего пространства::

        from pathlib import Path

        from PySide2.QtCore import QStandardPaths

        from axipy import Workspace

        path = Path(QStandardPaths.writableLocation(QStandardPaths.DocumentsLocation))
        path = path / "ws_out.mws"
        Workspace().save_file(path)
        print("Файл рабочего набора сохранен.", path)

        # Результат:
        # Файл рабочего набора сохранен. C:\\Users\\User\\Documents\\ws_out.mws
    """

    def __init__(self) -> None:
        self._shadow = ShadowWorkspace(gui_instance._shadow)

    def load_file(self, file_name: Union[str, Path]) -> None:
        """
        Читает из файла рабочего пространства и заносит данные в текущее окружение.
        
        Args:
            file_name: Имя входного файла.
        """
        if isinstance(file_name, Path):
            file_name = str(file_name)
        self._shadow.load_file(file_name)

    def save_file(self, file_name: Union[str, Path], overwrite: bool = False) -> None:
        """
        Сохраняет текущее состояние в файл рабочего пространства.

        Args:
            file_name: Имя выходного файла.
            overwrite: Перезаписывать существующий файл. По умолчанию False.
        """
        if isinstance(file_name, Path):
            file_name = str(file_name)

        if overwrite:
            self._shadow.save_file(file_name)
        else:
            if os.path.exists(file_name):
                raise FileExistsError("Файл уже существует")
            else:
                self._shadow.save_file(file_name)

    def load_string(self, workspace_str: str):
        """
        Читает из строки описание рабочего пространства и заносит данные в текущее окружение.
        
        Args:
            workspace_str: Строка с контентом рабочего пространства.
        """

        self._shadow.load_string(workspace_str)

    def save_string(self) -> str:
        """
        Сохраняет текущее состояние рабочего пространства в виде строки.

        Return:
            Строка с контентом рабочего пространства.
        """
        return self._shadow.save_string()
    
