import warnings
from typing import Union, Optional, Any

from axipy.cpp_core_core import StateObserver, DefaultKeys as _ShadowDefaultKeys, ValueObserver

from axipy._internal._shadow_instance_factory import _shadow_manager

__all__ = [
    "DefaultKeys",
    "StateManager",
    "ValueObserver",
    "state_manager"
]


class _DeprecMeta(type(_ShadowDefaultKeys)):

    def __getattribute__(cls, item):
        if cls is DefaultKeys:
            name = "axipy.DefaultKeys"
        elif cls is StateManager:
            name = "axipy.StateManager"
        else:
            raise RuntimeError("Unexpected class")
        msg = f"Class '{name}' is deprecated, use class 'axipy.ObserverManager' instead."
        warnings.warn(msg, DeprecationWarning, stacklevel=2)
        return super().__getattribute__(item)


class DefaultKeys(_ShadowDefaultKeys, metaclass=_DeprecMeta):
    # backwards compatibility
    Widget = _ShadowDefaultKeys.ActiveView
    MapView = _ShadowDefaultKeys.ActiveMapView
    TableView = _ShadowDefaultKeys.ActiveTableView


class StateManager(DefaultKeys):
    """Наблюдатели за состоянием.

    Note:
        Используйте готовый экземпляр этого класса :attr:`axipy.state_manager`.
    """

    def __getattribute__(self, item):
        if item != "__class__":
            msg = f"'axipy.state_manager' is deprecated, use class 'axipy.ObserverManager' instead."
            warnings.warn(msg, DeprecationWarning, stacklevel=2)
        return super().__getattribute__(item)

    @property
    def instance(self) -> StateObserver:
        return _shadow_manager.state_observer

    def get(self, id: Union[str, DefaultKeys]) -> ValueObserver:
        """Возвращает наблюдатель с указанным идентификатором.

        Args:
            id: Идентификатор.
        Raises:
            RuntimeError: Если наблюдатель с указанным идентификатором не найден.

        .. literalinclude:: /../../tests/doc_examples/test_example_value_observer.py
            :caption: Пример использования
            :pyobject: get_observer
            :lines: 2-
            :dedent: 4

        """
        id = self._to_name(id)
        return self.instance.get(id)

    def find(self, id: Union[str, DefaultKeys]) -> Optional[ValueObserver]:
        """Ищет наблюдатель с указанным идентификатором. Возвращает искомый
        наблюдатель или None, если не найдено.

        Args:
            id: Идентификатор.
        """
        id = self._to_name(id)
        return self.instance.find(id)

    def create(self, id: str, init_value: Any = None) -> ValueObserver:
        """Создает наблюдатель за значением.

        Args:
            id: Идентификатор.
            init_value: Первоначальное значение.
        Raises:
            RuntimeError: Если наблюдатель с указанным идентификатором уже
                существует.

        .. literalinclude:: /../../tests/doc_examples/test_example_value_observer.py
            :caption: Пример использования
            :pyobject: create_observer_deps
            :lines: 2-
            :dedent: 4

        .. literalinclude:: /../../tests/doc_examples/test_example_value_observer.py
            :caption: Пример наблюдателя за количеством открытых растров
            :pyobject: create_user_observer
            :lines: 2-
            :dedent: 4
        """
        return self.instance.create(id, init_value)

    def _to_name(self, key: Union[str, DefaultKeys, None]) -> str:
        if key is None or type(key) is str:
            return key
        return DefaultKeys.names()[key]


state_manager = StateManager()
