from typing import Any, Union, List, Optional

from axipy.cpp_core_dp import ShadowFeature as _ShadowFeature, ShadowFeatureIterator as _ShadowFeatureIterator

from .geometry import Geometry
from .style import Style

__all__ = [
    "GEOMETRY_ATTR",
    "STYLE_ATTR",
    "Feature"
]

GEOMETRY_ATTR = '+geometry'
"""Имя геометрического атрибута."""
STYLE_ATTR = '+style'
"""Имя атрибута со стилем."""


class Feature:
    """Запись в таблице.

    Работа с записью похожа на работу со словарем :any:`dict`. Но также
    допускает обращение по индексу.

    .. literalinclude:: /../../tests/doc_examples/da/test_example_feature.py
        :caption: Примеры.
        :pyobject: test_run_example_feature
        :lines: 2-
        :dedent: 4

    Args:
        properties: Значения атрибутов.
        geometry: Геометрия.
        style: Стиль.
        id: Идентификатор.
        **kwargs: Значения атрибутов.

    Note:
        Для доступа к геометрическому атрибуту и стилю по наименованию можно использовать предопределенные идентификаторы 
        `+geometry` и `+style` соответственно:

        * GEOMETRY_ATTR=+geometry
        * STYLE_ATTR=+style

    """

    def __init__(self, properties: dict = dict(), geometry: Geometry = None, style: Style = None, id: int = None,
                 **kwargs):
        self._shadow = _ShadowFeature({**properties, **kwargs})
        if geometry is not None:
            self.geometry = geometry
        if style is not None:
            self.style = style
        elif self.geometry:
            self.style = Style.for_geometry(geometry)
        if id is not None:
            self.id = id

    @classmethod
    def _wrap(cls, shadow: _ShadowFeature):
        result = cls.__new__(cls)
        result._shadow = shadow
        return result

    @property
    def id(self) -> int:
        """Идентификатор записи в таблице.

        Несохраненные записи в таблице будут иметь отрицательное значение.

        See Also:
            :attr:`Table.is_modified`, :meth:`Table.commit`

        Returns:
            0 если идентификатор не задан.
        """
        return self._shadow.id()

    @id.setter
    def id(self, value: int):
        self._shadow.setId(value)

    def __getitem__(self, index: Union[int, str]) -> Any:
        """Возвращает значение атрибута по индексу или ключу.

        Args:
            index: Индекс атрибута или его имя.
        """
        if isinstance(index, int):
            if index not in self:
                raise IndexError('Index out of bounds')
            index = self._shadow.indexToKey(index)
        elif isinstance(index, str):
            if index not in self:
                raise KeyError(f"Key '{index}' not found")
        if index == GEOMETRY_ATTR:
            return Geometry._wrap(_ShadowFeature.geometry(self._shadow))
        if index == STYLE_ATTR:
            return Style._wrap(_ShadowFeature.style(self._shadow))
        result = self._shadow[index]
        return result

    def __setitem__(self, index: Union[int, str], value: Any):
        """Присваивает атрибуту значение по индексу или ключу.

        Args:
            index: индекс атрибута или его имя.
            value: Присваиваемое значение.
        """
        if isinstance(index, int):
            if index not in self:
                raise IndexError('Index out of bounds')
            index = self._shadow.indexToKey(index)
        if index == GEOMETRY_ATTR:
            self._set_geometry(value)
            return
        if index == STYLE_ATTR:
            self._set_style(value)
            return
        self._shadow[index] = value

    def get(self, key: str, default: Any = None):
        """Возвращает значение заданного атрибута.

        Args:
            key: Имя атрибута.
            default: Значение по умолчанию.

        Returns:
            Искомое значение, или значение по умолчанию, если заданный
            атрибут отсутствует.
        """
        return self[key] if key in self else default

    @property
    def geometry(self) -> Optional[Geometry]:
        """Геометрия записи.

        See Also:

            :meth:`Feature.has_geometry`, :attr:`GEOMETRY_ATTR`

        Returns:
            Значение геометрического атрибута; или None, если значение
            пустое или отсутствует.
        """
        if GEOMETRY_ATTR not in self:
            return None
        return self[GEOMETRY_ATTR]

    @geometry.setter
    def geometry(self, value: Optional[Geometry]):
        self._set_geometry(value)

    def has_geometry(self) -> bool:
        """Проверяет, имеет ли запись атрибут с геометрией."""
        return self._shadow.hasGeometry()

    @property
    def style(self) -> Optional[Style]:
        """Стиль записи.

        See Also:

            :meth:`Feature.has_style`, :attr:`STYLE_ATTR`

        Returns:
            Значение атрибута со стилем; или None, если значение
            пустое или отсутствует.
        """
        if STYLE_ATTR not in self:
            return None
        return self[STYLE_ATTR]

    @style.setter
    def style(self, value: Optional[Style]):
        self._set_style(value)

    def has_style(self) -> bool:
        """Проверяет, имеет ли запись атрибут со стилем."""
        return self._shadow.hasStyle()

    def _set_geometry(self, value: Optional[Geometry]):
        if value is None:
            self._shadow[GEOMETRY_ATTR] = value
            return
        self._shadow.setGeometry(value._shadow)

    def _set_style(self, value: Optional[Style]):
        if value is None:
            self._shadow[STYLE_ATTR] = value
            return
        self._shadow.setStyle(value._shadow)

    def __contains__(self, key: Union[int, str]):
        """Проверяет наличие атрибута у записи.

        Args:
            key: Имя или номер атрибута.
        """
        return self._shadow.contains(key)

    def __len__(self) -> int:
        """Возвращает количество атрибутов в записи."""
        return self._shadow.__len__()

    def to_geojson(self) -> dict:
        """Представляет запись в виде, похожем на 'GeoJSON'."""
        result = {'type': 'Feature'}
        if self.has_geometry():
            bbox = self.geometry.bounds
            result["bbox"] = [bbox.xmin, bbox.ymin, bbox.xmax, bbox.ymax]
            result['geometry'] = self.geometry.to_geojson()
        if self.has_style():
            result['style'] = self.style.to_mapinfo()
        result['properties'] = {item[0]: item[1] for item in self.items() if item[0] not in [GEOMETRY_ATTR, STYLE_ATTR]}
        return result

    def keys(self) -> List[str]:
        """Возвращает список имен атрибутов."""
        return self._shadow.keys()

    def values(self) -> List:
        """Возвращает список значений атрибутов."""
        return [self[key] for key in self._shadow.keys()]

    def items(self) -> List[tuple]:
        """Возвращает список пар имя - значение.
        """
        keys = self.keys()
        return [(key, self[key]) for key in keys]

    def __eq__(self, other: 'Feature') -> bool:
        if not isinstance(other, Feature):
            return False
        if len(self) != len(other):
            return False
        other_keys = other.keys()
        for key in self.keys():
            if key in other_keys:
                continue
            return False
        for key in self.keys():
            if self[key] == other[key]:
                continue
            return False
        return True


class _FeatureIterator:
    def __init__(self, shadow: _ShadowFeatureIterator):
        self._shadow = shadow

    def __iter__(self):
        return self

    def __next__(self) -> Feature:
        return Feature._wrap(self._shadow.__next__())
