import importlib.metadata
from pathlib import Path

from axipy.utils import get_dependencies_folder

from .axparser.dependencies import PipDependencyInstaller, Dependencies

__all__ = [
    "PackageDependency",
]


class PackageDependency:
    """
    Сервисный класс проверки факта установки пакетов, а также установки новых пакетов.

    .. literalinclude:: /../../tests/doc_examples/utl/test_example_package.py
        :caption: Проверка наличия пакета и поиск файла в нем.
        :pyobject: test_run_example_query_package
        :lines: 2-
        :dedent: 4
    """

    @staticmethod
    def check_package_installed(name: str) -> dict:
        """
        Производит проверку на существование пакета

        Args:
            name: Наименование пакета или каталог, в котором расположен файл со списком requirements.txt

        Returns:
            dict, где ключ - это имя пакета, а значение - его путь. Если путь не найден, то значение будет None
        """

        def get_paths(n):
            try:
                dist = importlib.metadata.distribution(n)
                return {n: dist.locate_file('.')}
            except importlib.metadata.PackageNotFoundError:
                return {n: None}

        res = {}
        deps = Dependencies.find(Path(name))
        if deps is not None:
            for p in deps.packages():
                res.update(get_paths(p))
        else:
            res = get_paths(name)
        return res

    @staticmethod
    def list_package_files(name: str) -> list:
        """
        Возвращаем список файлов пакетов

        Args:
            name: Наименование пакета или каталог, в котором расположен файл со списком requirements.txt
        """

        def files(n):
            try:
                dist = importlib.metadata.distribution(n)
                return [dist.locate_file(f) for f in dist.files]
            except importlib.metadata.PackageNotFoundError:
                return []

        res = []
        deps = Dependencies.find(Path(name))
        if deps is not None:
            for p in deps.packages():
                res.update(files(p))
        else:
            res = files(name)
        return res

    @staticmethod
    def install_dependences(name: str):
        """
        Производит установку пакетов

        Args:
            name: Наименование пакета или каталог, в котором расположен файл со списком requirements.txt
        """
        dependencies_destination = get_dependencies_folder()
        deps_installer = PipDependencyInstaller()
        deps = Dependencies.find(Path(name))
        if deps is not None:
            deps_installer.install(deps, dependencies_destination)
        else:
            deps_installer.install_by_name(name, dependencies_destination)
