from .data_provider import DataProvider, DataObject
from .source import Source, Destination, Schema
from typing import List, Optional


class WmsSource(Source):
    pass


class WmsDataProvider(DataProvider):
    """Провайдер для Web Map Service.
    
    Note:
        Ссылку на провайдер можно получить через глобальную переменную :attr:`axipy.da.provider_manager.wms`.
    """
    _identifier = 'WmsDataProvider'

    def get_source(self, url_capabilities: str, layers: List[str], image_format: str = 'image/png', prj: str = None, style: str = None, alias: str = None) -> Source:
        """Создает источник данных.

        Args:
            url_capabilities: URL с метаданными capabilities.
            layers: Перечень слоев в виде списка.
            prj: Строка Системы Координат
            image_format: Формат выходного растра.
            style: Наименование стиля оформления.
            alias: Псевдоним для открываемого источника данных.
        """
        return WmsSource(
            Source._provider(self.id),
            Source._alias(alias),
            Source._prj(prj),
            {
                'src': url_capabilities,
                'layers' : layers,
                'imageFormat': image_format,
                'style' : style
            }
        )

    def open(self, url_capabilities: str, layers: List[str], image_format: str = 'image/png', prj: str = None, style: str = None, alias: str = None) -> DataObject:
        """Открывает объект данных.

        Args:
            url_capabilities: URL с метаданными capabilities.
            layers: Перечень слоев в виде списка.
            prj: Строка Системы Координат
            image_format: Формат выходного растра.
            style: Наименование стиля оформления.
            alias: Псевдоним для открываемого источника данных.

        Пример::

            wms_raster = provider_manager.wms.open('http://www.mapinfo.com/miwms', ['World'], prj='EPSG:4326', style='AreaStyleGreen')
        """
        return self.get_source(url_capabilities, layers, image_format, prj, style, alias).open()

    def get_destination(self):
        """
        Attention:
            Не поддерживается.

        Raises:
            NotImplementedError
        """
        raise NotImplementedError
    
    def create_open(self):
        """
        Attention:
            Не поддерживается.

        Raises:
            NotImplementedError
        """
        raise NotImplementedError
