from .data_provider import DataProvider, Table
from typing import List
from .source import Source, Destination, Schema
from ..attribute_schema import Schema
from axipy.cpp_core_dp import Converter
from axipy.cs import CoordSystem


class TabSource(Source):
    pass


class TabDestination(Destination):
    pass


class TabDataProvider(DataProvider):
    """Провайдер MapInfo.
    
    Note:
        Ссылку на провайдер можно получить через глобальную переменную :attr:`axipy.da.provider_manager.tab`.
    """
    _identifier = 'TabDataProvider'

    def get_source(self, filepath: str, alias: str = None) -> Source:
        """Создает источник данных.

        Args:
            filepath: Путь к файлу.
        """
        return TabSource(
            Source._provider(self.id),
            Source._table_file(filepath),
            Source._alias(alias),
        )

    def open(self, filepath: str, alias: str = None) -> Table:
        """Открывает объект данных.

        Args:
            filepath: Путь к файлу.
            alias: Псевдоним для открываемой таблицы.
        """
        return self.get_source(filepath, alias).open()

    def get_destination(self, filepath: str, schema: Schema) -> Destination:
        """Создает назначение объекта данных.

        Args:
            filepath: Путь к файлу.
            schema: Схема таблицы.
        """
        return TabDestination(schema,
                              Source._provider(self.id),
                              Source._table_file(filepath),
                              )

    def create_open(self, filepath: str, schema: Schema) -> Destination:
        """Создает и открывает объект данных.

        Args:
            filepath: Путь к файлу.
            schema: Схема таблицы.
        """
        return self.get_destination(filepath, schema).create_open()

    def __check_tab_file_suffix(self, filepath):
        from os import path
        _, ext =  path.splitext(filepath)
        if ext.lower() != '.tab':
            raise ValueError(f"Файл '{filepath}' должен иметь расширение 'tab'")

    def __check_tab_file_exists(self, filepath):
        from os import path
        if not path.exists(filepath):
            raise FileNotFoundError(f"Файл '{filepath}' не существует")
        self.__check_tab_file_suffix(filepath)


    def change_coordsystem(self, filepath: str, coordsystem: CoordSystem):
        """Изменяет координатную систему в TAB файле без изменения самих данных. Меняется непосредственно 
        сам файл, так что рекомендуется сделать копию.

        Args:
            filepath: Путь к файлу TAB (имя файла).
            coordsystem: Новое значение СК

        Raises:
            RuntimeError: При возникновении ошибки

        .. literalinclude:: /../../tests/doc_examples/test_example_dp.py
            :caption: Пример использования
            :pyobject: test_run_example_change_cs
            :lines: 2, 4-
            :dedent: 4
        """
        self.__check_tab_file_exists(filepath)
        convert_data = {
            "openWith": self.id,
            "src": filepath,
            "dest_prj": coordsystem.prj
        }
        Converter.convertTabToTab(convert_data)

    def remove_table_files(self, filepath: str):
        """Удаляет все связанные файлы с данным файлом в файловой системе.

        Args:
            filepath: Путь к файлу TAB (имя файла).

        Raises:
            RuntimeError: При возникновении ошибки

        .. literalinclude:: /../../tests/doc_examples/test_example_dp.py
            :caption: Пример использования
            :pyobject: test_run_example_remove_files
            :lines: 2,4-
            :dedent: 4

        """
        self.__check_tab_file_exists(filepath)
        convert_data = {
            "openWith": self.id,
            "src": filepath,
        }
        Converter.removeTabFiles(convert_data)


    def copy_table_files(self, src_filepath: str, dest_filepath: str):
        """Копирует все связанные файлы с данным файлом в файловой системе под новым именем.

        Args:
            src_filepath: Путь к исходному файлу TAB (имя файла).
            dest_filepath: Путь к выходному файлу TAB (имя файла).

        Raises:
            RuntimeError: При возникновении ошибки

        .. literalinclude:: /../../tests/doc_examples/test_example_dp.py
            :caption: Пример использования
            :pyobject: test_run_example_copy_files
            :lines: 2,3,6-
            :dedent: 4
        """
        self.__check_tab_file_exists(src_filepath)
        self.__check_tab_file_suffix(dest_filepath)
        from os import path
        if path.exists(dest_filepath):
            raise FileExistsError(f"Файл '{dest_filepath}' уже существует")

        convert_data = {
            "openWith": self.id,
            "src": src_filepath,
            "dest": dest_filepath
        }
        Converter.copyTabFiles(convert_data)

    def rename_table_files(self, src_filepath: str, dest_filepath: str):
        """Переименовывает файл и все связанные файлы с ним.

        Args:
            src_filepath: Путь к исходному файлу TAB (имя файла).
            dest_filepath: Путь к новому имени файла TAB (имя файла). Файл не должен существовать.

        Raises:
            RuntimeError: При возникновении ошибки

        .. literalinclude:: /../../tests/doc_examples/test_example_dp.py
            :caption: Пример использования
            :pyobject: test_run_example_rename_files
            :lines: 2,3,6-
            :dedent: 4
        """
        self.copy_table_files(src_filepath, dest_filepath)
        self.remove_table_files(src_filepath)
