
import enum
from typing import NamedTuple, Optional

from axipy.cpp_core_geometry import (ShadowStyle, ShadowPointStyle, ShadowPointCompatStyle, ShadowPointFontStyle,
                                     ShadowPointPictureStyle, ShadowLineStyle, ShadowFillStyle,
                                     ShadowPolygonStyle, ShadowTextStyle, ShadowCollectionStyle)
from axipy.da.Geometry import Geometry
from PySide2.QtGui import QColor, QPainter
from PySide2.QtCore import Qt


class Style:
    """Абстрактный класс стиля оформления геометрического объекта.

    Определяет как будет отрисован геометрический объект.

    Note:
        Для получения текстового представления стиля можно воспользоваться функцией :class:`str`.

    """

    def __init__(self):
        if type(self) is Style:
            raise NotImplementedError

    @classmethod
    def from_mapinfo(cls, mapbasic_string: str) -> 'Style':
        """Получает стиль из строки формата MapBasic.

        Args:
            mapbasic_string: Строка в формате MapBasic.

        .. literalinclude:: /../../tests/doc_examples/test_example_style.py
            :pyobject: test_run_example_style_from_mi
            :lines: 2
            :dedent: 4

        """
        result = ShadowStyle.from_mapinfo(mapbasic_string)
        if result is None or result.isEmpty():
            return None
        return Style._wrap(result)

    def to_mapinfo(self) -> str:
        """Возвращает строковое представление в формате MapBasic.

        .. literalinclude:: /../../tests/doc_examples/test_example_style.py
            :pyobject: test_run_example_style_from_mi
            :lines: 3-
            :dedent: 4
        """
        return self.shadow.to_mapinfo().strip()

#    def to_ogr(self) -> str:
#        """Возвращает строковое представление в формате OGR."""
#        return self.shadow.to_ogr().strip()

    def __eq__(self, other) -> bool:
        return other is not None and self.shadow.isEqual(other.shadow)

    @classmethod
    def for_geometry(cls, geom: Geometry) -> 'Style':
        """Возвращает стиль по умолчанию для переданного объекта.

        Args:
            geom: Геометрический объект, для которого необходимо получить соответствующий ему стиль.
        """
        return Style._wrap(ShadowStyle.for_geometry(geom.shadow))

    def draw(self, geometry: Geometry, painter: QPainter):
        """Рисует геометрический объект с текущим стилем в произвольном контексте вывода. Это может быть востребовано при 
        желании отрисовать геометрию со стилем на форме или диалоге.

        Args:
            geometry: Геометрия. Должна соответствовать стилю. Т.е. если объект полигон, а стиль для рисования точечных
                объектов, то ничего нарисовано не будет.
            painter: Контекст вывода.

        .. literalinclude:: /../../tests/doc_examples/test_example_style.py
            :caption: Пример отрисовки в растре и сохранение результата в файле.
            :pyobject: test_run_example_style_draw
            :lines: 7-
            :dedent: 4

        """
        self.shadow.draw(geometry.shadow, painter)

    def clone(self) -> 'Style':
        """
        Создаёт копию объекта стиля
        """
        return Style._wrap(self.shadow.clone())

    def _assign_shadow(self, shadow):
        self.shadow = shadow

    @classmethod
    def _wrap_typed(cls, shadow):
        result = cls.__new__(cls)
        result.shadow = shadow
        return result

    @classmethod
    def _wrap(cls, shadow):
        if shadow is None:
            return None
        if isinstance(shadow, ShadowPointCompatStyle):
            return PointCompatStyle._wrap_typed(shadow)
        elif isinstance(shadow, ShadowPointFontStyle):
            return PointFontStyle._wrap_typed(shadow)
        elif isinstance(shadow, ShadowPointPictureStyle):
            return PointPictureStyle._wrap_typed(shadow)
        elif isinstance(shadow, ShadowPointStyle):
            return PointStyle._wrap_typed(shadow)
        elif isinstance(shadow, ShadowLineStyle):
            return LineStyle._wrap_typed(shadow)
        elif isinstance(shadow, ShadowPolygonStyle):
            return PolygonStyle._wrap_typed(shadow)
        elif isinstance(shadow, ShadowFillStyle):
            return FillStyle._wrap_typed(shadow)
        elif isinstance(shadow, ShadowTextStyle):
            return TextStyle._wrap_typed(shadow)
        elif isinstance(shadow, ShadowCollectionStyle):
            return CollectionStyle._wrap_typed(shadow)
        else:
            return Style._wrap_typed(shadow)

    def __str__(self) -> str:
        return self.to_mapinfo()


class PointStyle(Style):
    """Стиль оформления точечных объектов.

    По умолчанию создается стиль на базе шрифта True Type, а параметры аналогичны значениям по умолчанию
    в методе :meth:`create_mi_font`.

    Поддерживается 3 вида оформления:

    * Совместимое с MapInfo версии 3. Для создания такого стиля необходимо использовать :meth:`create_mi_compat`.
    * На базе шрифтов True Type. Задано по умолчанию. Стиль создается посредством :meth:`create_mi_font`.
    * На базе растрового файла. Стиль можно создать с помощью :meth:`create_mi_picture`.

    .. literalinclude:: /../../tests/doc_examples/test_example_style.py
        :caption: Пример.
        :pyobject: test_run_example_style_point
        :lines: 2-
        :dedent: 4
    """

    def __init__(self):
        super().__init__()
        self._assign_shadow(ShadowPointStyle())

    @staticmethod
    def create_mi_compat(symbol: int = 35, color: QColor = Qt.red, pointSize: int = 8) -> 'PointCompatStyle':
        """Создание стиля в виде совместимого с MapInfo 3 :class:`PointCompatStyle`.

        Args:
            symbol: Номер символа, который будет отображен при отрисовке. Для создания невидимого символа используйте значение 31. Стандартный набор условных
                    знаков включает символы от 31 до 67,
            color: Цвет символа
            pointSize: Целое число, размер символа в пунктах от 1 до 48.
        """
        return Style._wrap(ShadowPointStyle.create_mi_compat(symbol, color, pointSize))

    @staticmethod
    def create_mi_font(symbol: int = 36, color: QColor = Qt.red, size: int = 8,
                       fontname: str = 'Axioma MI MapSymbols', fontstyle: int = 0, rotation: float = 0.0) -> 'PointFontStyle':
        """Создание стиля на базе шрифта True Type :class:`PointFontStyle`.

        Args:
            symbol: Целое, имеющее значение 31 или больше, определяющее, какой используется символ из шрифтов TrueType. Для создания невидимого символа
                    используйте значение 31.
            color: Цвет символа
            pointSize: Целое число, размер символа в пунктах от 1 до 48;
            fontname: Строка с именем шрифта TrueType (например, значение по умолчание 'Axioma MI MapSymbols')
            fontstyle: Стиль дополнительного оформления, например, курсивный текст. Возможные параметры см. в таблице ниже. Для указания нескольких параметров 
                       их суммируют между собой.
            rotation: Угол поворота символа в градусах.
        """
        return Style._wrap(ShadowPointStyle.create_mi_font(symbol, color, size, fontname, fontstyle, rotation))

    @staticmethod
    def create_mi_picture(filename: str, color: QColor = Qt.black, size: int = 12, customstyle: int = 0) -> 'PointPictureStyle':
        """Создание стиля с ссылкой на растровый файл :class:`PointPictureStyle`.

        Args:
            filename: Наименование растрового файла. Строка до 31 символа длиной. Данный файл должен находится в каталоге CustSymb с ресурсами. Например, `“Arrow.BMP”`.
            color: Цвет символа.
            size: Размер символа в в пунктах от 1 до 48.
            customstyle:  Задание дополнительных параметров стиля оформления. 
        """
        return Style._wrap(ShadowPointStyle.create_mi_picture(filename, color, size, customstyle))

    @property
    def color(self) -> QColor:
        """Цвет символа."""
        return self.shadow.get_color()
    
    @color.setter
    def color(self, c: QColor):
        self.shadow.set_color(c)


class PointCompatStyle(PointStyle):
    """Класс стиля, совместимого с MapInfo 3.

    Args:
        symbol: Номер символа, который будет отображен при отрисовке. Для создания невидимого символа используйте значение 31. Стандартный набор условных
                знаков включает символы от 31 до 67,
        color: Цвет символа
        pointSize: Целое число, размер символа в пунктах от 1 до 48.

    В системе доступны следующие стили:

    .. image:: /../images/style/style_point_mi3.png

    .. literalinclude:: /../../tests/doc_examples/test_example_style.py
        :caption: Пример.
        :pyobject: test_run_example_style_point_compat
        :lines: 2-
        :dedent: 4
    """
    def __init__(self, symbol: int = 35, color: QColor = Qt.red, pointSize: int = 8):
        super().__init__()
        self._assign_shadow(ShadowPointStyle.create_mi_compat(symbol, color, pointSize))

    @property
    def size(self) -> int:
        """Размер символа в пунктах."""
        return self.shadow.get_point_size()
    
    @size.setter
    def size(self, s: int):
        self.shadow.set_point_size(s)

    @property
    def symbol(self) -> int:
        """Номер символа."""
        return self.shadow.get_symbol()
    
    @symbol.setter
    def symbol(self, s: int):
        self.shadow.set_symbol(s)


class PointFontStyle(PointStyle):
    """Стиль на базе шрифта True Type.

    Args:
        symbol: Целое, имеющее значение 31 или больше, определяющее, какой используется символ из шрифтов TrueType. Для создания невидимого символа
                используйте значение 31.
        color: Цвет символа
        pointSize: Целое число, размер символа в пунктах от 1 до 48;
        fontname: Строка с именем шрифта TrueType (например, значение по умолчание 'Axioma MI MapSymbols')
        fontstyle: Стиль дополнительного оформления, например, курсивный текст. Возможные параметры см. в таблице ниже. Для указания нескольких параметров 
                    их суммируют между собой.
        rotation: Угол поворота символа в градусах.

    .. csv-table:: Возможные значения параметра `fontstyle`:
        :header: Значение, Наименование
        :widths: 10, 50

        0, Обычный текст
        1, Жирный текст
        16, Черная кайма вокруг символа
        32, Тень
        256, Белая кайма вокруг символа

    В системе доступны следующие стили:

    .. image:: /../images/style/style_point_mi_tt.png

    .. literalinclude:: /../../tests/doc_examples/test_example_style.py
        :caption: Пример.
        :pyobject: test_run_example_style_point_font
        :lines: 2-
        :dedent: 4
    """
    def __init__(self, symbol: int = 36, color: QColor = Qt.red, size: int = 8, fontname: str = 'Axioma MI MapSymbols', fontstyle: int = 0, rotation: float = 0.0):
        super().__init__()
        self._assign_shadow(ShadowPointStyle.create_mi_font(symbol, color, size, fontname, fontstyle, rotation))

    @property
    def symbol(self) -> int:
        """Номер символа."""
        return self.shadow.get_symbol()
    
    @symbol.setter
    def symbol(self, s: int):
        self.shadow.set_symbol(s)

    @property
    def size(self) -> int:
        """Размер символа в пунктах."""
        return self.shadow.get_point_size()
    
    @size.setter
    def size(self, s: int):
        self.shadow.set_point_size(s)

    @property
    def font_name(self) -> str:
        """Наименование шрифта."""
        return self.shadow.get_font_name()
    
    @font_name.setter
    def font_name(self, s: str):
        self.shadow.set_font_name(s)

    @property
    def rotation(self) -> float:
        """Угол поворота."""
        return self.shadow.rotation()
    
    @rotation.setter
    def rotation(self, s: float):
        self.shadow.set_rotation(s)

    @property
    def bold(self) -> bool:
        """Жирный шрифт.."""
        return self.shadow.get_bold()
    
    @bold.setter
    def bold(self, v: bool):
        self.shadow.set_bold(v)

    @property
    def has_shadow(self) -> bool:
        """Признак тени."""
        return self.shadow.get_shadow()
    
    @has_shadow.setter
    def has_shadow(self, v: bool):
        self.shadow.set_shadow(v)

    _BlackOutline = 16
    _WhiteOutline = 256

    @property
    def black_border(self) -> bool:
        """Темная окантовка."""
        return self.shadow.get_outline() == self._BlackOutline
    
    @black_border.setter
    def black_border(self, v: bool):
        self.shadow.set_outline(self._BlackOutline if v else 0)

    @property
    def white_border(self) -> bool:
        """Светлая окантовка."""
        return self.shadow.get_outline() == self._WhiteOutline
    
    @white_border.setter
    def white_border(self, v: bool):
        self.shadow.set_outline(self._WhiteOutline if v else 0)


class PointPictureStyle(PointStyle):
    """Стиль со ссылкой на растровый файл.

    Args:
        filename: Наименование растрового файла. Строка до 31 символа длиной. Данный файл должен находится в каталоге CustSymb с ресурсами. Например, `“Arrow.BMP”`.
        color: Цвет символа.
        size: Размер символа в в пунктах от 1 до 48.
        customstyle:  Задание дополнительных параметров стиля оформления. 

    .. csv-table:: Возможные значения параметра `customstyle`:
        :header: Значение, Наименование
        :widths: 10, 70

        0, "Флажки Фон и Покрасить одним цветом не установлены. Символ показывается стандартно. Все белые точки изображения становятся прозрачными и под ними видны объекты Карты."
        1, "Установлен флажок Фон; все белые точки изображения становятся непрозрачными."
        2, "Установлен флажок Покрасить одним цветом все не белые точки изображения красятся в цвет символа."
        3, "Установлены флажки Фон и Покрасить одним цветом."


    .. literalinclude:: /../../tests/doc_examples/test_example_style.py
        :caption: Пример.
        :pyobject: test_run_example_style_point_picture
        :lines: 2-
        :dedent: 4
    """
    def __init__(self, filename: str, color: QColor = Qt.black, size: int = 12, customstyle: int = 0):
        super().__init__()
        self._assign_shadow(ShadowPointStyle.create_mi_picture(filename, color, size, customstyle))

    @property
    def size(self) -> int:
        """Размер символа в пунктах."""
        return self.shadow.get_point_size()
    
    @size.setter
    def size(self, s: int):
        self.shadow.set_point_size(s)

    @property
    def filename(self) -> str:
        """Наименование файла растра."""
        return self.shadow.get_filename()
    
    @filename.setter
    def filename(self, s: str):
        self.shadow.set_filename(s)

    @property
    def show_background(self) -> bool:
        """Непрозрачный фон."""
        return self.shadow.get_show_background()
    
    @show_background.setter
    def show_background(self, b: bool):
        self.shadow.set_show_background(b)

    @property
    def apply_color(self) -> bool:
        """Применить цвет."""
        return self.shadow.get_apply_color()
    
    @apply_color.setter
    def apply_color(self, b: bool):
        self.shadow.set_apply_color(b)

    @property
    def actual_size(self) -> bool:
        """Реальный размер."""
        return self.shadow.get_actual_size()
    
    @actual_size.setter
    def actual_size(self, b: bool):
        self.shadow.set_actual_size(b)


class LineStyle(Style):
    """Стиль линейного объекта, совместимый с MapInfo.

    Args:
        pattern: Тип линии. Типы линий обозначаются кодами от 1 до 118. Тип 1 представляет собой невидимую линию.
        color: Цвет линии
        width: Толщина линии. Задается числом от 0 до 7, при этом линия нулевой ширины невидима на экране.
               11-2047 - это значения, которые могут быть преобразованы в пункты:
               ширина линии = (число пунктов * 10) + 10
               Значение 0 допустимо только для типа линии 1 или невидимых линий.

    В системе доступны следующие стили линии:

    .. image:: /../images/style/style_line.png

    .. literalinclude:: /../../tests/doc_examples/test_example_style.py
        :caption: Пример.
        :pyobject: test_run_example_style_line
        :lines: 2-
        :dedent: 4
    """

    def __init__(self, pattern: int = 2, color: QColor = Qt.black, width: int = 1):
        super().__init__()
        self._assign_shadow(ShadowLineStyle(pattern, color, width))

    @property
    def color(self) -> QColor:
        """Цвет линии."""
        return self.shadow.get_color()
    
    @color.setter
    def color(self, c: QColor):
        self.shadow.set_color(c)

    @property
    def pattern(self) -> int:
        """Номер стиля линии."""
        return self.shadow.get_pattern()

    @pattern.setter
    def pattern(self, id: int):
        self.shadow.set_pattern(id)

    @property
    def width(self) -> int:
        """Толщина линии."""
        return self.shadow.get_width()

    @width.setter
    def width(self, w: int):
        self.shadow.set_width(w)


class FillStyle(Style):
    """Стиль заливки полигонов :class:`PolygonStyle`.

    Args:
        pattern: Номер стиля заливки.
        color: Цвет основной заливки.

    В системе доступны следующие стили заливки:

    .. image:: /../images/style/style_fill.png
    """
    def __init__(self, pattern: int = 1, color: QColor = Qt.transparent):
        super().__init__()
        self._assign_shadow(ShadowFillStyle(pattern, color))

    @property
    def color(self) -> QColor:
        """Цвет линии."""
        return self.shadow.get_color()
    
    @color.setter
    def color(self, c: QColor):
        self.shadow.set_color(c)

    @property
    def bg_color(self) -> QColor:
        """Цвет фона. """
        return self.shadow.get_bg_color()

    @bg_color.setter
    def bg_color(self, c: QColor):
        self.shadow.set_bg_color(c)

    @property
    def pattern(self) -> int:
        """Номер стиля заливки."""
        return self.shadow.get_pattern()

    @pattern.setter
    def pattern(self, id: int):
        self.shadow.set_pattern(id)


class PolygonStyle(Style):
    """Стиль площадного объекта. По умолчанию создается прозрачный стиль :class:`FillStyle` с черной окантовкой :class:`LineStyle`.

    .. literalinclude:: /../../tests/doc_examples/test_example_style.py
        :caption: Пример.
        :pyobject: test_run_example_style_poly
        :lines: 2-
        :dedent: 4

    Args:
        pattern: Номер стиля заливки.
        color: Цвет основной заливки.
        pattern_pen: Цвет обводки. По умолчанию обводка отсутствует.
    """

    def __init__(self, pattern: int = 1, color: QColor = Qt.white, pattern_pen: int = 1):
        super().__init__()
        self._assign_shadow(ShadowPolygonStyle(pattern, color, pattern_pen))

    def set_pen(self, pattern: int = 2, color: QColor = Qt.black, width: int = 1):
        """Задание стиля обводки. Параметры аналогичны при задании стиля линии :meth:`LineStyle`

        Args:
            pattern: Номер стиля линии.
            color: Цвет линии
            width: Толщина линии.
        """
        self.shadow.set_pen(pattern, color, width)

    def set_brush(self, pattern: int = 1, color: QColor = Qt.white, bgColor: QColor = Qt.transparent):
        """Задание стиля заливки площадного объекта.

        Args:
            pattern: Номер стиля заливки. Шаблон задается числом от 1 до 71, при этом в шаблоне с номером 1 оба цвета отсутствуют,
                     а в шаблоне 2 отсутствует цвет фона. Шаблоны с кодами 9-11 зарезервированы для внутренних
                     целей.
            color: Цвет основной заливки.
            bgColor: Цвет заднего фона, если заливка неполная.

        Доступные стили заливки см. :class:`FillStyle`:
        """
        self.shadow.set_brush(pattern, color, bgColor)

    @property
    def border(self) -> LineStyle:
        """Стиль окантовки. Может отсутствовать. Для переопределения можно также использовать
        :meth:`set_pen`
        """
        return Style._wrap(self.shadow.get_border())

    @property
    def fill(self) -> FillStyle:
        """Стиль заливки."""
        return Style._wrap(self.shadow.get_fill())


class TextBackgroundType(enum.Enum):
    """
    Тип отрисовки фона текстового объекта.

    .. csv-table:: Значения
        :header: Значение, Наименование

        NoBackground, Фон отсутствует
        Outline, Обводка по контору букв текста
        Frame, Рамка вокруг текста
    """
    NoBackground = 0
    Outline = 1
    Frame = 2


class TextStyleEffects(NamedTuple): 
    """
    Эффекты текстового объекта

    Args:
        bold: Жирный
        italic: Курсив
        shadow: Тень
        spacing: Разрядка
        underline: Подчеркивание
        capital: Все заглавные

    """
    bold: bool = False
    italic: bool = False
    shadow: bool = False
    spacing: bool = False
    underline: bool = False
    capital: bool = False


class TextStyle(Style):
    """Стиль текстового объекта.

    Args:
        fontname: Наименование шрифта.
        size: Размер шрифта в пунктах. Может принимать значение 0 для подписей в окне карты, так как они являются атрибутами карты и их размер определяется динамически.
        style: Дополнительные параметры стиля. Подробнее см. в таблице ниже.
        color: Цвет шрифта
        backcolor: Цвет заднего фона, если он задан.

     .. csv-table:: Возможные значения параметра style
        :header: Значение, Наименование
        :widths: 10, 70

            0, Обычный
            1, Жирный
            2, Курсив
            4, Подчеркнутый
            16, Контур (только для `Macintosh`)
            32, Тень
            256, Кайма
            512, Капитель
            1024, Разрядка
    """

    def __init__(self, fontname: str, size: int, style: int = 0, forecolor: QColor = Qt.black, backcolor: QColor = Qt.transparent):
        super().__init__()
        self._assign_shadow(ShadowTextStyle(
            fontname, size, style, forecolor, backcolor))
        
    @property
    def color(self) -> QColor:
        """ Цвет текста """
        return self.shadow.color()

    @property
    def bg_color(self) -> QColor:
        """ Цвет фона текста """
        return self.shadow.bg_color()

    @property
    def fontname(self) -> str:
        """ Шрифт """
        return self.shadow.font_family()

    @property
    def effects(self) -> TextStyleEffects:
        """ Эффекты применяемые к текстовому объекту. """
        eff = self.shadow.effects()
        return TextStyleEffects(eff.bold, eff.italic, eff.shadow, \
            eff.spacing, eff.underline, eff.capital)

    @property
    def bg_type(self) -> TextBackgroundType:
        """ Тип отрисовки фона текста. """
        type_code = self.shadow.bg_type()
        if type_code == TextBackgroundType.NoBackground.value:
            return TextBackgroundType.NoBackground
        elif type_code == TextBackgroundType.Outline.value:
            return TextBackgroundType.Outline
        elif type_code == TextBackgroundType.Frame.value:
            return TextBackgroundType.Frame
        raise NotImplementedError()




class CollectionStyle(Style):
    """Смешанный стиль для разнородного типа объектов.

    Данный стиль представляет собой контейнер стилей. может применяться в купе с геометрическим объектом
    типа разнородная коллекция :class:`axipy.da.GeometryCollection`.
    Для задания или переопределения стилей простейших объектов, необходимо вызывать соответствующие методы для необходимых типов объектов.

    Note:
        Объекты стилей, полученные через методы :meth:`line`, :meth:`polygon` и т.д. 
        будут удалены сразу же после удаления объекта стиля коллекции. Если их нужно 
        сохранить, воспользуйтесь операцией :meth:`~axipy.da.Style.clone`.

    """

    def __init__(self):
        super().__init__()
        self._assign_shadow(ShadowCollectionStyle())

    def for_point(self, style: PointStyle):
        """Задание стиля для точечных объектов :class:`PointStyle`.  
        """
        self.shadow.for_point(style.shadow)

    def for_line(self, style: LineStyle):
        """Задание стиля для линейных объектов :class:`LineStyle`.  
        """
        self.shadow.for_line(style.shadow)

    def for_polygon(self, style: PolygonStyle):
        """Задание стиля для полигональных объектов :class:`PolygonStyle`.  
        """
        self.shadow.for_polygon(style.shadow)

    def for_text(self, style: TextStyle):
        """Задание стиля для текстовых объектов :class:`TextStyle`.  
        """
        self.shadow.for_text(style.shadow)

    @property
    def text(self) -> Optional[TextStyle]:
        """Стиль для текстовых объектов :class:`TextStyle`.  
        """
        return Style._wrap(self.shadow.text())

    @property
    def line(self) -> Optional[LineStyle]:
        """Стиль для линейных объектов :class:`LineStyle`.  
        """
        return Style._wrap(self.shadow.line())

    @property
    def polygon(self) -> Optional[PolygonStyle]:
        """Стиль для полигональных объектов :class:`PolygonStyle`.  
        """
        return Style._wrap(self.shadow.polygon())

    @property
    def point(self) -> Optional[PointStyle]:
        """Стиль для точечных объектов :class:`PointStyle`.  
        """
        return Style._wrap(self.shadow.point())

    def find_style(self, geom: Geometry) -> Style:
        """Пытаемся найти стиль подходящий для переданной геометрии
        """
        return Style._wrap(self.shadow.style(geom.shadow))
