import types
from axipy.cpp_app import *
from axipy.da import DataManager, DataObject
from axipy.gui import View, gui_instance
from axipy.gui.ViewWrapper import MapView
from axipy.render import Layer
from PySide2.QtCore import QObject, Qt, QUrl
from PySide2.QtWidgets import QMainWindow, QMdiSubWindow, QWidget, QDockWidget
from PySide2.QtGui import QIcon
from typing import Optional


class MainWindow(QObject):
    """Главное окно ГИС Аксиома.

    Note:
        Используйте готовый объект :attr:`axipy.app.mainwindow`.
    """

    def __init__(self):
        super().__init__()
        self.shadow = None
        self._catalog = None

    @staticmethod
    def show() -> 'MainWindow':
        """Создает и показывает главное окно программы."""
        ShadowMainWindow.create(gui_instance.shadow)
        mainwindow.qt_object().show()
        return mainwindow

    def service(self):
        if self.shadow is None:
            from axipy.app import mainwindow_instance_hidden
            self.shadow = mainwindow_instance_hidden
        return self.shadow

    @property
    def is_valid(self) -> bool:
        """Корректность состояния главного окна."""
        try:
            self.service()
            return True
        except ImportError:
            return False

    def qt_object(self) -> QMainWindow:
        """Возвращает Qt5 объект окна."""
        return self.service().qt_object()

    @property
    def catalog(self) -> DataManager:
        """Хранилище объектов приложения.

        Это то же хранилище, которое отображается в панели "Открытые данные".

        Note:
            При открытии объектов данных :meth:`axipy.da.ProviderManager.openfile`
            они автоматически попадают в каталог.
        """
        if self._catalog is None:
            self._catalog = DataManager._wrap(gui_instance.shadow.catalog())
        return self._catalog

    def add(self, view: View) -> QMdiSubWindow:
        """Добавляет окно просмотра данных.

        Args:
            view: окно просмотра данных.

        Note:
            При создании окон просмотра данных :meth:`axipy.gui.ViewManager.create_mapview` или :meth:`axipy.gui.ViewManager.create_tableview`
            они автоматически добавляются в главное окно программы.
        """
        return self.service().add(view.shadow)

    def load_workspace(self, fileName: str):
        """Читает рабочее пространство из файла.
        
        Args:
            fileName: Наименование входного файла.
        """
        self.service().open_workspace(fileName)

    def save_workspace(self, fileName: str):
        """Сохраняет рабочее пространство в файл.

        Args:
            fileName: Наименование выходного файла.
        """
        self.service().save_workspace(fileName)

    def add_layer_interactive(self, layer: Layer)-> MapView:
        """Добавляет слой с запросом на помещение на текущую карту или в новую."""
        return View._wrap(self.service().addLayerInteractive(layer.shadow))

    def add_layer_current_map(self, layer: Layer)-> MapView:
        """Добавляет слой в текущей карте."""
        return View._wrap(self.service().addLayerCurrentMap(layer.shadow))

    def add_layer_new_map(self, layer: Layer)-> MapView:
        """Открывает слой в новой карте."""
        return View._wrap(self.service().addLayerNewMap(layer.shadow))

    def add_dock_widget(self, dock_widget: QDockWidget, area: Qt.DockWidgetArea, icon: QIcon = None) -> bool: 
        """Добавляет панель в главное окно приложения. При успешном добавлении возвращает True. Если же данная панель уже присутствует,
        то команда игнорируется и возвращается False.
        Элементы управления, которые требуется разместить на панели, создаются в дополнительном окне, а уже это окно, в свою очередь,
        устанавливается для панели (см. пример ниже).
        
        Args:
            dock_widget: Пользовательская созданная панель.
            area: Расположение.
            icon: Иконка для отображения в списке всех доступных панелей.

        Пример::

            from PySide2.QtWidgets import QDockWidget, QWidget, QPushButton
            from PySide2.QtCore import Qt

            dock = QDockWidget('Заголовок')
            widget = QWidget()
            layout = QVBoxLayout()
            button = QPushButton("Кнопка")
            button.clicked.connect(lambda: print('Реакция на кнопку'))
            layout.addWidget(button)
            layout.addStretch()
            widget.setLayout(layout)
            dock.setWidget(widget)
            app.mainwindow.add_dock_widget(dock, Qt.RightDockWidgetArea, QIcon('filename.png'))
        """
        return self.service().addDockWidget(dock_widget, area, icon if icon is not None else QIcon())

    def remove_dock_widget(self, dock: QDockWidget):
        """Удаляет существующую панель у главного окна приложения."""
        return self.service().removeDockWidget(dock)

    def show_html_url(self, url: QUrl, caption: Optional[str]):
        """Показывает окно для локального файла html или если это web страница, запускает браузер по ассоциации

        Args:
            url: Ссылка на файл html или адрес страницы.
            caption: Заголовок окна
        """
        self.service().showHtmlUrl(caption, url)


mainwindow = MainWindow()


def _decorate_opener():
    from axipy.da.providers import opener_instance as op
    op.shadow.setDefaultCatalog(mainwindow.catalog.shadow)


def _apply_post(original_method, post_func):
    """Takes method and returns another  method that calls it while calling
    post_func at the end.

    post_func takes the instance itself and result as parameters.
    """
    def _wrapper_method(self, *args, **kwargs):
        results = original_method(*args, **kwargs)
        post_func(self, results)
        return results
    return _wrapper_method


def _decorate_view_manager():
    from axipy.gui import view_manager as service
    def _show_view(instance, view):
        if isinstance(view, View):
            window = mainwindow.add(view)
            window.show()
    service.create_mapview = types.MethodType(_apply_post(service.create_mapview, _show_view), service)
    service.create_tableview = types.MethodType(_apply_post(service.create_tableview, _show_view), service)
    service.create_reportview = types.MethodType(_apply_post(service.create_reportview, _show_view), service)
    service.create_legendview = types.MethodType(_apply_post(service.create_legendview, _show_view), service)


def _hook_core_registered():
    from axipy.da import provider_manager, data_manager
    provider_manager._init_instance()
    data_manager._init_instance(gui_instance._cpp_catalog())


def _hook_mainwindow_registered():
    """Вызывается из С++ после регистрации главного окна."""
    _decorate_opener()
    _decorate_view_manager()


__all__ = [
    'MainWindow',
    '_hook_mainwindow_registered',
    '_hook_core_registered',
    'mainwindow',
]
