"""
Точка входа в модуль.
"""
import traceback
import uuid
from pathlib import Path

from PySide2.QtWidgets import QMessageBox
from axipy import AxiomaPlugin, Position, tr
from osgeo import gdal

from .helper import debug, clear_data_dir
from .wizardconverter import WizardConverter


class Plugin(AxiomaPlugin):

    def load(self) -> None:

        self._action = self.create_action(
            title=tr("Конвертер"),
            on_click=self.show_plugin_window,
            icon=self.local_file("icon", "Translate-01.png"),
            tooltip=tr("Конвертер векторных форматов"),
            doc_file="index.html"
        )

        position = Position(tr("Основные"), tr("Команды"))
        position.add(self._action)

        self._plugin_window = None

        self._uuid = uuid.uuid4().hex  # type: str

        clear_data_dir(Path(self.user_plugin_data_dir()))

    def unload(self) -> None:
        self._action.remove()

        self._plugin_window = None

    def show_plugin_window(self) -> None:
        """
        Функция показа окна плагина.
        """

        if self._plugin_window is None:
            success = self.init_plugin_window()
            if not success:
                return None

        # Запомнить состояние режима исключений gdal при показе окна
        self._plugin_window.use_exceptions = gdal.GetUseExceptions()
        self._plugin_window.show()

    def init_plugin_window(self) -> bool:
        """
        Создание окна плагина
        """
        try:
            self._plugin_window = WizardConverter(
                self.window(), self.language, Path(self.user_plugin_data_dir(self._uuid)))
        except Exception:
            traceback.print_exc()

            m_box = QMessageBox(
                QMessageBox.Warning,  # icon
                tr("Ошибка"),  # title
                tr("Не удалось создать окно Помощника."),  # text
                buttons=QMessageBox.Ok,
                parent=self.window(),
            )
            m_box.exec_()

            self._plugin_window = None
            return False
        else:
            return True
