import importlib
import sys
import traceback
from enum import Enum
from pathlib import Path
from typing import Optional, Callable, List

from PySide2.QtCore import Qt, QLocale
from PySide2.QtGui import QDoubleValidator, QMouseEvent, QPaintEvent, QPainter, QKeyEvent
from PySide2.QtWidgets import QWidget, QComboBox, QCheckBox, QGroupBox, QVBoxLayout, QLabel, \
    QSpinBox, QLineEdit, QRadioButton, QFormLayout, QFileDialog, QPushButton
from axipy import tr, MapView, MapTool, SelectToolBehavior, Point, VectorLayer, ActiveToolPanel, MultiPolygon, \
    TypeSqlDialect
from axipy.app import Notifications
from axipy.concurrent import ProgressSpecification, AxipyProgressHandler, task_manager, ProgressGuiFlags
from axipy.da import Table, Feature, Style
from axipy.da import provider_manager, data_manager
from axipy.gui import view_manager, selection_manager
from axipy.sql import geometry_uid

from .. import helper

is_master = str(Path(__file__).parents[1].name).endswith("master")
if is_master:
    for module in (helper,):
        importlib.reload(module)

# Минимальное количество точек для построения полигонов Вороного
MIN_POINTS_COUNT = 2


class MakePolygonsMode(Enum):
    """
    Режим построения полигонов
    """
    NewTable = 0  # Создать новую таблицу
    TmpTable = 1  # Создать новую временную таблицу
    EditableLayer = 2  # Построить на редактируемом слое


class VoronoiSettings:
    """ Параметры, собираемые виджетом """

    def __init__(self) -> None:
        self.inp_layer = None  # type: Optional[int]

        self.polygons_mode = MakePolygonsMode.TmpTable  # type: MakePolygonsMode

        self.file_name = None

        # Значения по умолчанию взяты из документации к SpatiaLite
        self.edges_only = False  # type: bool
        self.frame_extra_size = 5  # type: int
        self.tolerance = 0  # type: float

    def __str__(self) -> str:
        txt = [
            f"self.inp_layer_index = {self.inp_layer}",
            f"self.polygons_mode = {self.polygons_mode}",
            f"self.file_name = {self.file_name}"
        ]
        return "\n".join(txt)


class VoronoiWidget(QWidget):

    def __init__(self, iface, title) -> None:
        super().__init__()
        self.iface = iface
        self.title = title

        self.layer_ids = None
        self._settings = VoronoiSettings()

        self._init_gui()
        self.refresh_layers()

    def _init_gui(self) -> None:
        vbox = QVBoxLayout()

        txt = tr("Выбрать слой с точками")
        label = QLabel(txt)
        vbox.addWidget(label)
        self.cbox_choose_inp_table = QComboBox()
        vbox.addWidget(self.cbox_choose_inp_table)

        txt = tr("Режим построения полигонов")
        self.mode_g_box = QGroupBox(txt)
        g_box_vbox = QVBoxLayout()
        r1 = QRadioButton(tr("Создать новую таблицу"), objectName="r1")

        def create_file_dialog(qfile_dialog: Callable):
            lnd = QLineEdit(readOnly=True)

            btn = QPushButton(
                text="...", maximumWidth=20,
                clicked=lambda lnd_arg=lnd: lnd_arg.setText(qfile_dialog(filter="*.tab")[0])
            )

            hbox_arg = helper.create_hbox(lnd, btn)
            return hbox_arg

        hbox = create_file_dialog(QFileDialog.getSaveFileName)
        self.lnd_file_name = hbox.itemAt(0).widget()
        self.lnd_file_name.setText(self._settings.file_name)
        r2 = QRadioButton(tr("Создать временную таблицу"), objectName="r2")
        r3 = QRadioButton(tr("Построить на редактируемом слое"), objectName="r3")
        for w in (r1, r2, r3):
            g_box_vbox.addWidget(w)
        g_box_vbox.insertLayout(1, hbox)
        self.mode_g_box.setLayout(g_box_vbox)
        self.mode_g_box.findChildren(QRadioButton)[self._settings.polygons_mode.value].setChecked(True)
        vbox.addWidget(self.mode_g_box)

        txt = tr("Настройки полигонов Вороного")
        g_box = QGroupBox(txt)
        g_box_layout = QFormLayout()
        label = QLabel(tr("Расширить границу"))
        self.spin_box_frame_extra_size = QSpinBox(maximum=100, suffix="%")
        self.spin_box_frame_extra_size.setValue(self._settings.frame_extra_size)
        g_box_layout.addRow(label, self.spin_box_frame_extra_size)
        label = QLabel(tr("Чувствительность"))
        self.lnd_tolerance = QLineEdit()
        validator = QDoubleValidator(bottom=sys.float_info.min, top=sys.float_info.max, decimals=6)
        validator.setNotation(QDoubleValidator.StandardNotation)
        self.lnd_tolerance.setValidator(validator)
        sys_locale = QLocale.system()  # type: QLocale
        self.lnd_tolerance.setText(sys_locale.toString(self._settings.tolerance, prec=6))
        self.lnd_tolerance.setToolTip(tr(
            "Значение чувствительности в единицах измерения выбранного слоя.\n"
            "Если координаты точек отличаются друг от друга меньше чем на введенное значение, "
            "то они не будут учитываться при выполнении операции."))
        hbox = helper.create_hbox(label, self.lnd_tolerance)
        hbox.setContentsMargins(0, 0, 0, 0)
        w = QWidget(layout=hbox)
        g_box_layout.addRow(label, w)
        label = QLabel(tr("Только стороны"))
        self.check_box_edges_only = QCheckBox(objectName="edges_only")
        self.check_box_edges_only.setChecked(self._settings.edges_only)
        g_box_layout.addRow(label, self.check_box_edges_only)
        g_box.setLayout(g_box_layout)
        vbox.addWidget(g_box)

        vbox.addStretch()

        self.setLayout(vbox)

    def refresh_layers(self) -> None:

        self.cbox_choose_inp_table.clear()

        layer_ids = []

        n = selection_manager.count
        if n > 0:
            self.cbox_choose_inp_table.addItem(data_manager.selection.name)
            layer_ids.append("selection")

        active_view = view_manager.active
        if not isinstance(active_view, MapView):
            return None
        cosmetic_layer = active_view.map.cosmetic
        cosmetic_items = cosmetic_layer.data_object.items()
        if next(cosmetic_items, None) is not None:
            self.cbox_choose_inp_table.addItem(cosmetic_layer.title)
            layer_ids.append("cosmetic")

        layers = helper.active_map_vector_layers(cosmetic=False)
        if layers is None:
            return None
        layers_titles = list(map(lambda layer: layer.title, layers))
        layer_ids.extend(range(len(layers_titles)))

        self.layer_ids = layer_ids
        self.cbox_choose_inp_table.addItems(layers_titles)

    def read_settings(self) -> VoronoiSettings:
        result = self._settings

        result.file_name = self.lnd_file_name.text()

        current_index = self.cbox_choose_inp_table.currentIndex()
        result.inp_layer = self.layer_ids[current_index]

        tmp_list = self.mode_g_box.findChildren(QRadioButton)
        for i, elem in enumerate(tmp_list):
            if elem.isChecked():
                result.polygons_mode = MakePolygonsMode(i)

        result.edges_only = self.check_box_edges_only.isChecked()
        result.frame_extra_size = self.spin_box_frame_extra_size.value()
        result.tolerance = helper.text_to_float(self.lnd_tolerance.text(), self.lnd_tolerance.locale())

        return result


class MakePolygons:

    def __init__(self, title: str, settings: VoronoiSettings) -> None:
        self._title = title  # type: str
        self._settings = settings  # type: VoronoiSettings

        self._inp_table = None  # type: Optional[Table]
        self._out_table = None  # type: Optional[Table]

    def run(self) -> None:
        try:
            self._find_input_table()

            self._check_input_items()

            # Получение полигонов в потоке
            helper.ensure_sql_init_for_bg_thread()
            spec = ProgressSpecification(window_title=self._title, flags=ProgressGuiFlags.CANCELABLE)
            polygons = task_manager.run_and_get(spec, self._make_polygons_in_thread)
            if polygons is None:
                raise Exception("polygons is None")

            polygons_count = len(polygons)
            if polygons_count == 0:
                raise Exception("polygons count is 0")

            self._ensure_out_table()

            # Вставка в таблицу в потоке

            def insert_items(ph, items):
                try:
                    ph.prepare_to_write_changes()
                    self._out_table.insert(items)
                except Exception:
                    if is_master:
                        traceback.print_exc()
                    return False
                else:
                    return True

            result = task_manager.run_and_get(spec, insert_items, polygons)
            if not result:
                raise Exception("can't insert")

            # Попытка добавить полученные данные на активную карту
            try:
                if self._settings.polygons_mode in (MakePolygonsMode.NewTable, MakePolygonsMode.TmpTable):
                    active_view = view_manager.active
                    if isinstance(active_view, MapView) and self._out_table.is_spatial:
                        layer = VectorLayer.create(self._out_table)
                        active_view.map.layers.append(layer)
            except Exception:
                if is_master:
                    traceback.print_exc()
                Notifications.push(self._title, tr("Не удалось добавить полигоны Вороного на карту."),
                                   Notifications.Warning)

            Notifications.push(self._title, tr(f"Было добавленно записей: {polygons_count}"), Notifications.Success)

        except Exception:
            if is_master:
                traceback.print_exc()
            Notifications.push(self._title, tr("Не удалось построить полигоны Вороного."), Notifications.Critical)

    def _find_input_table(self) -> None:
        inp_table = None

        inp_layer = self._settings.inp_layer

        if inp_layer == "selection":
            selection = data_manager.selection
            if selection is not None:
                inp_table = selection

        elif inp_layer == "cosmetic":
            active_view = view_manager.active
            layer = active_view.map.cosmetic
            inp_table = layer.data_object

        elif isinstance(inp_layer, int):
            def get_vector_layer(i):
                # Существующая таблица
                active_view_arg = view_manager.active
                vector_layers = filter(lambda l: isinstance(l, VectorLayer), active_view_arg.map.layers)
                return list(vector_layers)[i].data_object

            inp_table = get_vector_layer(inp_layer)

        if inp_table is None:
            raise Exception("table is None")

        self._inp_table = inp_table

    def _check_input_items(self) -> None:
        n = self._inp_table.count()
        items = self._inp_table.items()

        def is_point(f: Feature):
            return isinstance(f.geometry, Point)

        if n < MIN_POINTS_COUNT or len(list(filter(is_point, items))) < MIN_POINTS_COUNT:
            Notifications.push(self._title, tr(f"Недостаточно точек. Минимальное колличество: {MIN_POINTS_COUNT}."))
            raise Exception("not enough points")

    def _make_voronov_multipolygon(self) -> MultiPolygon:
        """
        Создание мультиполигона Вороного
        """

        inp_table_name = self._inp_table.name

        edges_only = False  # Этот параметр обрабатывается отдельно
        frame_extra_size = self._settings.frame_extra_size
        tolerance = self._settings.tolerance
        # Основной запрос
        query_text = f"SELECT VoronojDiagram(" \
                     f"Collect(FromAxiGeo({geometry_uid})), {edges_only}, {frame_extra_size}, {tolerance}" \
                     f") " \
                     f"FROM {inp_table_name} " \
                     f"WHERE Str$({geometry_uid}) == 'Point'"

        table = None
        try:
            table = data_manager.query_hidden(query_text, TypeSqlDialect.sqlite)
            items = table.items()
        finally:
            if isinstance(table, Table):
                table.close()

        # В таблице одна Feature с возможной геометрией типа MultiPolygon
        feature = next(items, None)
        if feature is None or not feature.has_geometry():
            raise Exception("feature is None or no geom")

        g = feature.geometry
        if isinstance(g, MultiPolygon):
            return g
        else:
            raise Exception("geometry not MultiPolygon")

    def _make_polygons_in_thread(self, ph: AxipyProgressHandler) -> List[Feature]:
        ph.set_description(tr("Подготовка к созданию полигонов Вороного"))
        multipolygon = self._make_voronov_multipolygon()

        ph.set_description(tr("Создание полигонов Вороного"))
        inp_table = self._inp_table
        inp_cs = inp_table.coordsystem

        polygons_with_attr = []
        ph.set_max_progress(len(multipolygon))
        for polygon in multipolygon:
            ph.raise_if_canceled()
            polygon.coordsystem = inp_cs

            items_in_rect = inp_table.itemsInRect(polygon.bounds)
            for item in items_in_rect:
                g = item.geometry
                if isinstance(g, Point) and g.intersects(polygon):

                    if not self._settings.edges_only:
                        item.geometry = polygon
                        item.style = Style.from_mapinfo("Pen (1, 2, 0) Brush (1, 16777215, 16777215)")
                    else:  # Если только стороны
                        item.geometry = polygon.to_linestring()
                        item.style = Style.from_mapinfo("Pen (1, 2, 0)")

                    polygons_with_attr.append(item)
                    ph.add_progress(1)

        return polygons_with_attr

    def _ensure_out_table(self) -> None:
        inp_schema = self._inp_table.schema

        out_table = None
        if self._settings.polygons_mode == MakePolygonsMode.NewTable:
            out_table = provider_manager.tab.create_open(self._settings.file_name, inp_schema)
        elif self._settings.polygons_mode == MakePolygonsMode.TmpTable:
            out_table = provider_manager.tab.create_open("", inp_schema)
        elif self._settings.polygons_mode == MakePolygonsMode.EditableLayer:
            out_table = helper.ensure_editable_table()
        if out_table is None:
            raise Exception("out_table is None")
        else:
            self._out_table = out_table


class VoronoiMapTool(MapTool):

    def __init__(self, iface, title, observer_id):
        super().__init__()
        self.iface = iface
        self.title = title
        self.observer_id = observer_id

        self.tool_panel = None

        self.__connections = None

        self.__selector = SelectToolBehavior(self)

    def load(self):

        service = ActiveToolPanel()
        widget = VoronoiWidget(self, self.title)
        self.tool_panel = service.make_acceptable(
            title=self.title,
            observer_id=self.observer_id,
            widget=widget)

        w = widget

        self.__connections = [
            helper.Connection(self.tool_panel.accepted, self.tool_panel_accepted),
            helper.Connection(self.tool_panel.panel_was_closed, self.reset),
            helper.Connection(selection_manager.changed, w.refresh_layers),
            helper.Connection(view_manager.active_changed, self.active_changed),
            helper.Connection(self.view.map.need_redraw, self.active_changed),
        ]

        self.tool_panel.activate()

    def unload(self):

        self.__selector = None

        self.tool_panel.deactivate()

        for connection in self.__connections:
            connection.disconnect()

    def keyPressEvent(self, event: QKeyEvent) -> Optional[bool]:

        if self.__selector.keyPressEvent(event):
            return self.BlockEvent
        elif event.key() == Qt.Key_Escape:
            self.reset()
            return self.BlockEvent
        else:
            return super().keyPressEvent(event)

    def mouseMoveEvent(self, event: QMouseEvent) -> None:
        self.__selector.mouseMoveEvent(event)

    def mousePressEvent(self, event: QMouseEvent) -> Optional[bool]:
        return self.__selector.mousePressEvent(event)

    def mouseReleaseEvent(self, event: QMouseEvent) -> Optional[bool]:
        return self.__selector.mouseReleaseEvent(event)

    def paintEvent(self, event: QPaintEvent, painter: QPainter) -> None:
        self.__selector.paintEvent(event, painter)

    def tool_panel_accepted(self):

        settings = self.tool_panel.widget.read_settings()

        if settings.polygons_mode == MakePolygonsMode.EditableLayer:
            # TODO: Передавать таблицу в класс создания полигонов
            try:
                helper.ensure_editable_table()
            except Exception:
                Notifications.push(self.title, tr("Не удалось найти редактируемый слой."), Notifications.Critical)
                return None
        elif settings.polygons_mode == MakePolygonsMode.NewTable:
            if not Path(settings.file_name).is_absolute():
                Notifications.push(self.title, tr("Пожалуйста, выберите имя для новой таблицы."),
                                   Notifications.Critical)
                return None

        self.tool_panel.disable()

        make_polygons = MakePolygons(self.title, settings)
        make_polygons.run()

        self.tool_panel.try_enable()

    def active_changed(self):
        if view_manager.active and view_manager.active.widget != self.view.widget:
            return None

        self.tool_panel.activate()

        w = self.tool_panel.widget
        w.refresh_layers()

    def selection_manager_changed(self):
        if view_manager.active and view_manager.active.widget != self.view.widget:
            return None

        w = self.tool_panel.widget
        w.refresh_layers()
