from typing import Optional

import axipy
from axipy.da import Table, GeometryType, Feature, LineStyle, CollectionStyle, Style
from axipy.gui import view_manager
from axipy.interface import AxiomaInterface

from .strategy_base import SelectStrategy
from ..data_types import NamedSearchPatterns, SearchPatterns, SearchGroupPatterns
from ..widgets.select_by_style_widget import SelectByStyleWidget
from ..widgets.style_widget_interface import SelectByStyleWidgetInterface


class LineStrategy(SelectStrategy):
    """
    Стратегия для выделения линейных объектов
    """

    def __init__(self, f: Feature, table: Table) -> None:
        super().__init__(f, table)
        self.__geometries = [
            GeometryType.Line,
            GeometryType.Arc,
            GeometryType.LineString,
            GeometryType.MultiLineString,
            GeometryType.GeometryCollection
        ]
        self.__width = axipy.tr("Ширина")
        self.__pattern = axipy.tr("Паттерн линии")
        self.__color = axipy.tr("Цвет")
        self.__patterns = [self.__width, self.__pattern, self.__color]

    def identify(self, style: Style) -> bool:
        return isinstance(style, LineStyle)

    def fill_widget(self, widget: 'SelectByStyleWidget'):
        geometries = {gtype: True for gtype in self.__geometries}
        widget.search_geometries = geometries
        widget.search_patterns = NamedSearchPatterns(self.search_patterns(), \
                                                     self.string_pattern_group_key())

    def settings_widget(self, iface: AxiomaInterface) -> SelectByStyleWidgetInterface:
        widget = SelectByStyleWidget(view_manager.global_parent, iface)
        widget.put_content_on_top(True)
        self.fill_widget(widget)
        return widget

    def search_patterns(self) -> SearchPatterns:
        return {pattern: True for pattern in self.__patterns}

    def string_pattern_group_key(self) -> str:
        return 'line_key'

    def is_supported_style(self, target: LineStyle, source: LineStyle, group_patterns: SearchGroupPatterns) -> bool:
        assert (isinstance(target, LineStyle))
        patterns = group_patterns[self.string_pattern_group_key()]
        if patterns[self.__width] and \
                target.width != source.width:
            return False
        if patterns[self.__pattern] and \
                target.pattern != source.pattern:
            return False
        if patterns[self.__color] and \
                target.color != source.color:
            return False
        return True

    def extract_style_from_collection(self, collection_style: CollectionStyle) -> Optional[Style]:
        line_style = collection_style.line  # type: LineStyle
        if line_style:
            return line_style.clone()
        return line_style
