import os
from PyQt5.QtWidgets import QWidget, QFileDialog
from PyQt5.QtGui import QIcon
from PyQt5.QtCore import QStringListModel, Qt, QFileInfo, QCoreApplication, QDir
from PyQt5.uic import loadUi

from axioma.core import *
from axioma.core.dp import *
from .DoubleProgressWidget import DoubleProgressWidget
import traceback

class ExportToDatabaseWidget(QWidget):

    def __init__(self,  catalog, parent=None):
        super().__init__(parent)
        cwd = os.path.dirname(__file__)
        self.__ui = loadUi(os.path.join(cwd, "exportToDatabaseWidget.ui"), self)
        self.initUi()
        self.__catalog = catalog
        self.__catalog.updated.connect(self.onUpdatedCatalog)
        self.onUpdatedCatalog(None)

    def initUi(self):
      self.__ui.tbSelectFiles.setIcon(QIcon.fromTheme("open"))
      self.__ui.tbSelectFiles.clicked.connect(self.onOpenFilesClicked)
      self.__ui.tbDeleteSelected.setIcon(QIcon.fromTheme("delete"))
      self.__ui.tbDeleteSelected.clicked.connect(self.onDeleteSelectedClicked)
      self.__ui.listSourceTables.itemSelectionChanged.connect(self.onSelectionChanged)
      self.__ui.pbExport.clicked.connect(self.exportProcess)
      self.__ui.leErrorFileName.setText(QDir.temp().filePath('export_error.sql'))
      self.__ui.tbErrorFileName.clicked.connect(self.onChoiceErrorFileNameClicked)
      self.__ui.tbErrorFileName.setIcon(QIcon.fromTheme("open"))
      self.__ui.leLogFileName.setText(QDir.temp().filePath('export_log.sql'))
      self.__ui.tbLogFileName.setIcon(QIcon.fromTheme("open"))
      self.__ui.tbLogFileName.clicked.connect(self.onChoiceLogFileNameClicked)
      self.__ui.cbDestinations.currentIndexChanged.connect(self.onDestinationChanged)
      self.__ui.cbOwner.currentIndexChanged.connect(self.onOwnerChanged)

    def onDestinationChanged(self, index):
      self.__ui.cbOwner.clear()
      if self.__ui.cbDestinations.currentIndex() != -1:
        ds = self.__ui.cbDestinations.currentData()
        self.__ui.cbOwner.blockSignals(True)
        self.__ui.cbOwner.addItems(ds.metadata().listOwners())
        self.__ui.cbOwner.setCurrentText(ds.metadata().currentSchema())
        self.__ui.cbOwner.blockSignals(False)

    def onOwnerChanged(self, index):
      ds = self.__ui.cbDestinations.currentData()
      if (self.__ui.cbOwner.currentIndex() != -1 
          and ds.provider().class_id() == "OracleDataProvider"
          and self.__ui.cbOwner.currentText().lower() != ds.metadata().currentSchema().lower()):
        axioma.app.core.notificationManager().push(Notification('Ошибка', 'При экспорте в другую схему БД oracle создание пространственного индекса не поддерживается.', Notification.NR_Warning))

    def onUpdatedCatalog(self, reason):
      try:
        self.__ui.cbDestinations.clear()
        for ds in self.__catalog.allDataSources():
          if (isinstance(ds.provider(), DatabaseProvider)):
              def_ = JsonDefinition()
              if (ds.provider().canAcceptDestination(def_)):
                self.__ui.cbDestinations.addItem(ds.dataSourceDefinition().description(), ds);
        self.checkEnabledExport()
      except Exception as ex:
        print(ex)

    def keyPressEvent(self, event):
      super().keyPressEvent(event)
      if (event.key() == Qt.Key_Delete):
        onDeleteSelectedClicked()

    def onOpenFilesClicked(self):
      settings = axioma.app.core.settings()
      lastPath = settings.value(DefaultSettingsList.LastOpenPath)
      dirName = QFileDialog.getOpenFileNames(self, 'Выбор файлов для экспорта', lastPath, 'Файлы TAB (*.tab);;Все файлы (*.*)')
      if len(dirName[0]):
          settings.setValue(DefaultSettingsList.LastOpenPath, QFileInfo(dirName[0][0]).absolutePath())
          for fn in dirName[0]:
            if len(self.__ui.listSourceTables.findItems(fn, Qt.MatchExactly)) == 0:
              self.__ui.listSourceTables.addItem(fn)
          self.checkEnabledExport()

    def onSelectionChanged(self):
      self.__ui.tbDeleteSelected.setEnabled(len(self.__ui.listSourceTables.selectedItems()))

    def checkEnabledExport(self):
      enabled = self.__ui.listSourceTables.count() and self.__ui.cbDestinations.currentIndex() != -1
      self.__ui.pbExport.setEnabled(enabled)

    def onDeleteSelectedClicked(self):
      selected_items = self.__ui.listSourceTables.selectedItems()
      for item in selected_items:
        self.__ui.listSourceTables.takeItem(self.__ui.listSourceTables.row(item))
      self.checkEnabledExport()

    def onChoiceErrorFileNameClicked(self):
      fileName = QFileDialog.getSaveFileName(self, 'Файл ошибок', self.__ui.leErrorFileName.text(), 'Файлы ошибок (*.sql)')
      if fileName[0]:
          self.__ui.leErrorFileName.setText(fileName[0])

    def onChoiceLogFileNameClicked(self):
      fileName = QFileDialog.getSaveFileName(self, 'Файл лога команд', self.__ui.leLogFileName.text(), 'Файлы команд (*.sql)')
      if fileName[0]:
          self.__ui.leLogFileName.setText(fileName[0])

    def exportProcess(self):
      self.__ui.pbExport.setEnabled(False)
      try:
        params = {}
        if (self.__ui.cbGeomAsText.isChecked()):
          params[DatabaseProviderExportSupport.geometryAsTextTag] = True
        if (self.__ui.cbDropTable.isChecked()):
          params[DatabaseProviderExportSupport.dropTableTag] = True
        if (self.__ui.cbCreateIndex.isChecked()):
          params[DatabaseProviderExportSupport.createIndexTag] =  True
        if (self.__ui.cbRegisterMapcatalog.isChecked()):
          params[DatabaseProviderExportSupport.mapCatalogTag] = True
        if (self.__ui.gbErrorFileName.isChecked()):
          params[DatabaseProviderExportSupport.errorFileTag] = self.__ui.leErrorFileName.text()
        if (self.__ui.gbLogFileName.isChecked()):
          params[DatabaseProviderExportSupport.logFileTag] = self.__ui.leLogFileName.text()
        params[DatabaseProviderExportSupport.geometryColumnNameTag] = 'geometry'
#        print(params)
        cnt = self.__ui.listSourceTables.count()
        progress = DoubleProgressWidget(axioma.app.mainWindow)
        progress.setMaximumValueForTotalProgress(cnt)
        progress.setFixedSize(progress.width(), progress.height()) # disabled maximized button
        progress.move(axioma.app.mainWindow.rect().center() - progress.rect().center())
        progress.show()
        for idx in range(0, cnt):
          self.exportFile(self.__ui.listSourceTables.item(idx).text(), params, progress)
          progress.setCurrentValueForTotalProgress(idx+1)
          QCoreApplication.processEvents()
        progress.close()
      except Exception as ex:
        print(ex)
      self.__ui.pbExport.setEnabled(True)

    def onProgress(self, count, progress):
      if progress is not None:
        progress.setCurrentValueForCurrentProgress(count)
      QCoreApplication.processEvents()

    def exportFile(self, fileName, params, progress):
      print('Export:', fileName)
      ds = self.__ui.cbDestinations.currentData()
      try:
        table = axioma.core.open_json({'src': fileName})
        if not isinstance(table, Table):
          raise TypeError('Исходный файл {} не является таблицей.'.format(fileName))
        progress.setCurrentTableName(table.name())
        jd = JsonDefinition()
        def_ = ds.dataSourceDefinition()
        JsonDBExportController.setDbDef(jd, def_)
        if self.__ui.cbOwner.currentText().lower() == ds.metadata().currentSchema().lower():
          tableName = table.name()
        else:
          tableName = '{}.{}'.format(self.__ui.cbOwner.currentText(), table.name())
        JsonDBExportController.setObject(jd, tableName)
        JsonDBExportController.setPrj(jd, table.coordSystem())
        JsonDBExportController.setExportParams(jd, params)
        context = ProviderExportContext(ModifiedTableSchema(table.tableSchema()))
        cursor = SignalingCursor(table.select(QuerySelectAll( table.tableSchema().attributeNames())))
        cursor.moved.connect(lambda count: self.onProgress(count, progress))
        progress.destroyed.connect(lambda : cursor.abort())
        progress.setMaximumValueForCurrentProgress(table.count())
        ds.provider().exportFeatures(cursor, context, jd)
      except Exception as ex:
        print(ex)
        axioma.app.core.notificationManager().push(Notification('Ошибка', str(ex), Notification.NR_Warning))
