from typing import List, Dict
from .source import Source, Destination
from ..DataObjectWrapper import Table, DataObject


class DataProvider:
    """Абстрактный провайдер данных.

    """

    def __init__(self, info: Dict):
        if type(self) is DataProvider:
            raise NotImplementedError
        self.info = info

    @property
    def id(self) -> str:
        """Идентификатор провайдера."""
        return self.info['id']

    def file_extensions(self) -> List[str]:
        """Список поддерживаемых расширений файлов.

        Returns:
            Пустой список для не файловых провайдеров.
        """
        return self.info.get('extensions', [])

    def get_source(self) -> Source:
        """Создает источник данных.

        Raises:
            NotImplementedError: Если провайдер не поддерживает создание источников.
        """
        raise NotImplementedError

    def get_destination(self) -> Destination:
        """Создает назначение объекта данных.

        Raises:
            NotImplementedError: Если провайдер не поддерживает создание назначений.
        """
        raise NotImplementedError

    def open(self, *args, **kwargs):
        """Открывает объект данных.

        Пример::

            provider.open(...)

        Что эквивалентно::

            provider.get_source(...).open()

        See also:

            :meth:`DataProvider.source`.
        """
        return self.get_source(*args, **kwargs).open()

    def create(self, *args, **kwargs):
        return self.create_open(*args, **kwargs)

    def create_open(self, *args, **kwargs):
        """Создает и открывает объект данных.

        Пример::

            provider.create_open(...)

        Что эквивалентно::

            provider.get_destiantion(...).create_open()

        See also:

            :meth:`DataProvider.destination`.
        """
        return self.get_destination(*args, **kwargs).create_open()
