"""Модуль источников данных.

В данном модуле содержатся классы и методы для работы с источниками
данных.

Attributes:
    provider_manager (axipy.da.ProviderManager): Готовый экземпляр открытия/создания объектов данных.
    state_manager (axipy.da.StateManager): Готовый экземпляр наблюдателей за состоянием.
    data_manager (axipy.da.DataManager):
        Хранилище объектов приложения.

        Это то же хранилище, которое отображается в панели "Открытые данные".

        Note:
            При открытии объектов данных :meth:`axipy.da.ProviderManager.openfile`
            они автоматически попадают в каталог.


.. class:: DefaultKeys

    Идентификаторы наблюдателей по умолчанию.

    .. csv-table:: Атрибуты
        :header: Значение, Тип, Наименование

        Selection, :class:`bool`, Есть выборка
        Editable, :class:`bool`, Активная карта имеет редактируемый слой
        SelectionEditable, :class:`bool`, Карта имеет редактируемый слой и есть выделенные объекты на одном из слоев карты
        SelectionEditableIsSame, :class:`bool`, Карта имеет редактируемый слой и выборку на этом слое
        Widget, :class:`bool`, Есть активное окно
        MapView, :class:`bool`, Есть активное окно карты
        TableView, :class:`bool`, Есть активное окно таблицы
        HasTables, :class:`bool`, Открыта хотя бы одна таблица

.. class:: ValueObserver

    Наблюдатель за одним значением.

    .. method:: value()

        Возвращает значение.

        ::

            # Эквивалентно
            v = obs.value()
            v = obs()

        :rtype: :any:`typing.Any`

    .. method:: setValue(value)

        Устанавливает значение.
        
        При изменении значения испускается сигнал :attr:`changed`.

        :param value: Новое значение.
        :type value: :any:`typing.Any`

    .. method:: changed(value)
        :property:

        Сигнал об изменении значения.

        :param value: Новое значение.
        :type value: :any:`typing.Any`
        :rtype: :class:`PySide2.QtCore.Signal`

        ::
        
            def print_func(value):
                print(value)
        
            observer.changed.connect(print_func)
"""

from axipy._util import fixup_module_metadata
from axipy.cpp_core_geometry import Type as GeometryType
from PySide2.QtCore import Signal

from .FeatureWrapper import Feature, GEOMETRY_ATTR, STYLE_ATTR
from .DataObjectWrapper import DataObject, Table, Raster, GCP, SupportedOperations, QueryTable, SelectionTable, CosmeticTable
from .DataManagerWrapper import DataManager
from .attribute_schema import Attribute, Schema
from .attribute_factory import AttributeFactory, attr
from .Geometry import (Geometry, GeometryType as Type, Point, Line, LineString, Polygon,
                       GeometryCollection, MultiPoint, MultiLineString, MultiPolygon)
from .Style import (Style, PointStyle, PointCompatStyle, PointFontStyle, PointPictureStyle, 
                    LineStyle, FillStyle, PolygonStyle, TextStyle, TextBackgroundType, 
                    TextStyleEffects, CollectionStyle)
from .providers import (ProviderManager, DataProvider, Source, Destination,
                        CsvDataProvider, ExcelDataProvider, MifMidDataProvider,
                        ShapeDataProvider, SqliteDataProvider, TabDataProvider,
                        OracleDataProvider, PostgreDataProvider, MsSqlDataProvider,
                        RestDataProvider, GdalDataProvider, TmsDataProvider,
                        WmtsDataProvider, OgrDataProvider, SvgDataProvider, WmsDataProvider
                        )
from .sqldialect import TypeSqlDialect
from .TabFile import TabFile
from typing import Union, Optional, Any
from axipy.cpp_core_core import StateObserver, ValueObserver, DefaultKeys


class StateManager(DefaultKeys):
    """Наблюдатели за состоянием.
    
    Note:
        Используйте готовый экземпляр этого класса :attr:`axipy.da.state_manager`.
    """

    @property
    def instance(self) -> StateObserver:
        return state_instance

    def get(self, id: Union[str, DefaultKeys]) -> ValueObserver:
        """Возвращает наблюдатель с указанным идентификатором.

        Args:
            id: Идентификатор.
        Raises:
            RuntimeError: Если наблюдатель с указанным идентификатором не найден.

        .. literalinclude:: /../../tests/doc_examples/test_example_value_observer.py
            :caption: Пример использования
            :pyobject: get_observer
            :lines: 2-
            :dedent: 4

        """
        id = self._to_name(id)
        return self.instance.get(id)

    def find(self, id: Union[str, DefaultKeys]) -> Optional[ValueObserver]:
        """Ищет наблюдатель с указанным идентификатором. Возвращает искомый
        наблюдатель или None, если не найдено.

        Args:
            id: Идентификатор.
        """
        id = self._to_name(id)
        return self.instance.find(id)

    def create(self, id: str, init_value: Any = None) -> ValueObserver:
        """Создает наблюдатель за значением.

        Args:
            id: Идентификатор.
            init_value: Первоначальное значение.
        Raises:
            RuntimeError: Если наблюдатель с указанным идентификатором уже
                существует.

        .. literalinclude:: /../../tests/doc_examples/test_example_value_observer.py
            :caption: Пример использования
            :pyobject: create_observer_deps
            :lines: 2-
            :dedent: 4

        .. literalinclude:: /../../tests/doc_examples/test_example_value_observer.py
            :caption: Пример наблюдателя за количеством открытых растров
            :pyobject: create_user_observer
            :lines: 2-
            :dedent: 4
        """
        return self.instance.create(id, init_value)

    def _to_name(self, key: Union[str, DefaultKeys, None]) -> str:
        if key is None or type(key) is str:
            return key
        return DefaultKeys.names()[key]


state_manager = StateManager()


provider_manager = ProviderManager()
data_manager = DataManager._wrap(None)


# fix sphinx inheritance and typehints
fixup_module_metadata(__name__, globals())
del fixup_module_metadata
