from itertools import chain
from PySide2.QtCore import QRectF, Qt
from PySide2.QtWidgets import QInputDialog, QMessageBox
from axipy.cs import CoordTransformer
from axipy.da import Geometry, Type, Point
from axipy.da import GEOMETRY_ATTR
from axipy.render import VectorLayer
from axipy.gui import MapTool, view_manager

# Переопределенный класс инструмента
class MyTool(MapTool):

    def __init__(self, caption = None) -> None:
        super().__init__()
        self.caption = caption

    def load(self):
        self.attr_index = 0
    
    def mousePressEvent(self, event) -> bool:
        # Левая кнопка мыши - вывод значений
        if event.button() == Qt.LeftButton:
            self._show_info(event)
        # Правая кнопка мыши - выбор атрибута
        elif event.button() == Qt.RightButton:
            self._select_attribute_index()
        return self.PassEvent

    def mouseReleaseEvent(self, event) -> bool:
        """Blocks right click context menu."""
        return self.BlockEvent

    def _show_info(self, event) -> None:
        scene_box = self.get_select_rect(event.pos())
        tables = self.get_tables()
        feature_iters = map(lambda table: self.get_features(table, scene_box), tables)
        features = chain.from_iterable(feature_iters)
        # Вывод в формате таблица:значение по всем попавшим в выбор слоям
        text = '\n'.join(tab.name + ': ' + str(feature[self.attr_index]) for tab, feature in features)
        if text:
            QMessageBox.information(view_manager.global_parent, 'Информация', text)
    
    def _select_attribute_index(self):
        i, ok = QInputDialog.getInt(None, 'Параметры', self.caption,
            self.attr_index, 0)
        if ok:
            self.attr_index = i

    def get_tables(self):
        for layer in self.view.map.layers:
            if not isinstance(layer, VectorLayer):
                continue
            yield layer.data_object

    def get_features(self, table, scene_box):
        if table.coordsystem != self.view.coordsystem:
            transformer = CoordTransformer(self.view.coordsystem, table.coordsystem)
            scene_box = transformer.transform(scene_box)
        center = scene_box.center
        scene_point = Point(center.x, center.y, table.coordsystem)
        for feature in table.items(bbox=scene_box):
            geometry = feature[GEOMETRY_ATTR]
            if geometry.type == Type.Point or geometry.contains(scene_point):
                yield table, feature
