import itertools
import traceback
from pathlib import Path
from typing import List, Optional, Union, Iterator

from PySide2.QtCore import QLocale
from PySide2.QtSql import QSqlQuery
from PySide2.QtWidgets import QHBoxLayout
from axipy import Rect, VectorLayer
from axipy.concurrent import task_manager
from axipy.cs import CoordSystem
from axipy.da import Table, data_manager, Geometry
from axipy.gui import MapView, selection_manager, view_manager
from axipy.render import Layer
from axipy.sql import get_database

is_master = str(Path(__file__).parents[0].name).endswith("master")


def quick_envelope(geometries: List[Geometry]) -> Rect:
    """
    Функция возвращает нормализованный ограничивающий прямоугольник.
    """
    if len(geometries) == 0:
        return None
    target = geometries[0].bounds
    target.normalize()
    for geom in geometries:
        r = geom.bounds
        r.normalize()
        target.xmin = min(target.xmin, r.xmin)
        target.xmax = max(target.xmax, r.xmax)
        target.ymin = min(target.ymin, r.ymin)
        target.ymax = max(target.ymax, r.ymax)
    return target


class DataManagerGuard:
    """
    В конструкторе добавляем в DataManager, а в деструкторе удаляем.
    TODO: Правильный путь - это дать возможность добавлять таблицы, которые
    не отображаются в открытых данных. Очень удобно для временных таблиц
    """

    def __init__(self, table) -> None:
        from secrets import token_hex
        self._table = table
        self._table.name = 'selection_' + token_hex(nbytes=16)
        data_manager.add(self._table)

    def __del__(self):
        data_manager.remove(self._table)

    @property
    def data(self) -> Table:
        return self._table

    @property
    def name(self) -> str:
        return self._table.name


def wrap_tmp_table(table: Table):
    return DataManagerGuard(table)


def selection_as_tmp_table() -> DataManagerGuard:
    """
    Создаём временную таблицу на основе выборки, которую регистрируем в DataManager,
    чтобы её можно было использовать в SQL запросах.
    """
    table = selection_manager.get_as_table()
    return wrap_tmp_table(table)


def editable_table_from_view(view: MapView) -> Optional[Table]:
    if not isinstance(view, MapView):
        return None
    editable_layer = view.editable_layer
    if editable_layer is None:
        return None
    editable_table = editable_layer.data_object
    if not isinstance(editable_table, Table):
        print(f"Типом источника данных должен быть axipy.da.Table вместо {type(editable_table)}")
        return None
    return editable_table


def current_editable_table() -> Table:
    """
    Функция возвращает таблицу для текущего редактируемого слоя активной карты.
    """
    map_view = view_manager.active  # type: MapView
    return editable_table_from_view(map_view)


def ensure_editable_table() -> Table:
    table = current_editable_table()
    assert table is not None
    return table


def normalize_if_not_valid(
        geometries: Union[List[Geometry], Geometry]) -> Union[List[Geometry], Geometry]:
    def normalize(geometries: List[Geometry]):
        res = []
        for g in geometries:
            if not g.is_valid:
                res.append(g.normalize())
            else:
                res.append(g)
        return res

    if isinstance(geometries, list):
        return normalize(geometries)
    else:
        return normalize([geometries])[0]


def text_to_float(text: str, locale: QLocale) -> float:
    """
    Преобразовывает текстовою строку в float с использованием локали.

    RuntimeError при ошибках преобразования
    """
    value, ok = locale.toFloat(text)
    if not ok:
        raise RuntimeError("Не удалось преобразовать значение "
                           "поля ввода в число.")
    return value


def ensure_sql_init_for_bg_thread():
    query = QSqlQuery(get_database())
    task_manager.run_in_gui(lambda: query.exec_("Select 'Init SQL Engine'"))


def clone_cs(cs: CoordSystem) -> CoordSystem:
    if cs is None:
        return None
    return CoordSystem._wrap(cs.shadow)


class Connection:
    """
    Вспомогательный класс который хранит соединение между произвольным сигналом 
    и любой функцией переданной в виде слота
    """

    def __init__(self, signal, slot, auto_connect=True) -> None:
        self.__is_connected = False
        self.signal = signal
        self.slot = slot
        if auto_connect:
            self.connect()

    def __del__(self):
        try:
            self.disconnect()
        except Exception:
            if is_master:
                traceback.print_exc()

    def connect(self):
        if self.__is_connected:
            return
        self.signal.connect(self.slot)
        self.__is_connected = True

    def disconnect(self):
        if self.__is_connected:
            self.signal.disconnect(self.slot)
            self.__is_connected = False


def has_table(mv: MapView, table: Table) -> bool:
    """ 
    Проверяет есть ли таблица среди слоёв карты. 
    """
    return find_layer(mv, table) is not None


def find_layer(mv: MapView, table: Table) -> Optional[Layer]:
    """
    Ищет слой в карте содежащий переданну таблицу. 
    """
    if table is None:
        return None
    is_table = lambda l: isinstance(l.data_object, Table)
    for layer in filter(is_table, mv.map.layers):
        if layer.data_object.name == table.name:
            return layer
    if mv.map.cosmetic.data_object.name == table.name:
        return mv.map.cosmetic
    return None


def create_hbox(*widgets) -> QHBoxLayout:
    """
    Создает горизонтальную разметку с входными виджетами.

    :param widgets: Список виджетов.
    :return: Горизонтальная разметка (QHBoxLayout).
    """
    hbox = QHBoxLayout()
    for widget in widgets:
        hbox.addWidget(widget)
    return hbox


def active_map_vector_layers(cosmetic=True) -> Optional[Iterator[VectorLayer]]:
    active_view = view_manager.active
    if not isinstance(active_view, MapView):
        return None

    layers = active_view.map.layers
    vector_layers = filter(lambda x: isinstance(x, VectorLayer), layers)

    if cosmetic:
        vector_layers = itertools.chain((active_view.map.cosmetic,), vector_layers)

    return vector_layers
