import os
from dataclasses import dataclass
from PySide2.QtWidgets import QDialog, QFileDialog
from PySide2.QtGui import QIcon
from PySide2.QtUiTools import QUiLoader
from PySide2.QtCore import Qt

from .CoordinateProcess import *
from axipy import *


class TypeResultAction(enum.Enum):
    current_layer = 0
    new_table = 1

@dataclass
class LoadPointsLastUsedData:
    typeCoordinates: TypeCoordinatesData = TypeCoordinatesData.value
    is_closed: bool = True
    field_geom: str = None
    field_geom_part: str = None
    field_geom_point: str = None
    field_geom_value1: str = None
    field_geom_value2: str = None


class LoadPointsDialog(QDialog):

    NoneValue = ''
    __ResultAction = TypeResultAction.current_layer

    def __init__(self, iface, coordsystem, lastUsed) -> None:
        super().__init__(iface.window())
        self.__tr = iface.tr
        self.__last_used = lastUsed
        self._coordsystem = coordsystem
        uiFile = os.path.join(os.path.dirname(__file__), "LoadPointsDialog.ui")
        self.__ui = QUiLoader().load(uiFile, self)
        self.__load_ui()
        self.__ui.setParent(view_manager.global_parent)
        self.__update_button_text(coordsystem)
        self.__ui.setWindowFlags(self.windowFlags() & ~Qt.WindowContextHelpButtonHint)
        self.__out_filename = ''

        self.__ui.cbTypeData.setCurrentIndex(lastUsed.typeCoordinates)
        self.__ui.cbClosedAsPolygon.setCheckState(Qt.Checked if lastUsed.is_closed else Qt.Unchecked)

    def __load_ui(self):
        for t in data_manager.tables:
            if len(t.schema) >= 2:
                self.__ui.cbTables.addItem(t.name)
        self.__ui.cbTables.currentIndexChanged.connect(self.__tablesIndexChanged)
        self.__ui.cbTypeData.currentIndexChanged.connect(self.__typeDataIndexChanged)
        self.__ui.pushButtonSK.clicked.connect(self.__chooseCoordSystem)
        if self.__ui.cbTables.count():
            self.__tablesIndexChanged(0)
        self.__ui.pbExport.clicked.connect(self.__export)
        self.__ui.pbExport.setEnabled( self.__ui.cbTables.count())
        self.__ui.pbInsert.clicked.connect(self.__insert)
        can_insert = isinstance(view_manager.active, MapView) and view_manager.active.map.editable_layer is not None
        self.__ResultAction = TypeResultAction.current_layer if can_insert else TypeResultAction.new_table
        self.__ui.pbInsert.setEnabled(can_insert and self.__ui.cbTables.count())
        self.__ui.pbClose.clicked.connect(self.__close)

    def __acceptClose(self):
        self.__ui.close()
        self.__ui.setResult(QDialog.Accepted)

    def lastUsedValues(self):
        def f_fn(fn):
            return fn if fn != '' else None
        res = LoadPointsLastUsedData()
        res.typeCoordinates = self.result_type_coordinates()
        res.is_closed = self.result_is_closed_polygon()
        res.field_geom = f_fn(self.__ui.cbGeometryId.currentText())
        res.field_geom_part = f_fn(self.__ui.cbGeometryPartId.currentText())
        res.field_geom_point = f_fn(self.__ui.cbPointNumber.currentText())
        res.field_geom_value1 = f_fn(self.__ui.cbValue1.currentText())
        res.field_geom_value2 = f_fn(self.__ui.cbValue2.currentText())
        return res
    
    def __export(self):
        fn, _ = QFileDialog.getSaveFileName(self, self.__tr('Имя выходного файла'), filter = "MapInfo TAB (*.tab);;", 
                dir = Settings.value(Settings.LastSavePath))
        if len(fn) > 0:
            self.__out_filename = fn
            self.__ResultAction = TypeResultAction.new_table
            self.__acceptClose()

    def __insert(self):
        self.__ResultAction = TypeResultAction.current_layer
        self.__acceptClose()

    def __close(self):
        self.__ui.close()
        self.__ui.setResult(QDialog.Rejected)
    
    def __update_button_text(self, coordsystem):
        if coordsystem is not None:
            self.__ui.pushButtonSK.setText(coordsystem.description)

    def __chooseCoordSystem(self):
        dialog = ChooseCoordSystemDialog(self._coordsystem)
        if dialog.exec() == QDialog.Accepted:
            self._coordsystem = dialog.chosenCoordSystem()
            self.__update_button_text(self._coordsystem)

    def __typeDataIndexChanged(self, idx):
        if idx == 0:
            self.__ui.lblValue1.setText(self.__tr('Х координата:'))
            self.__ui.lblValue2.setText(self.__tr('Y координата:'))
        else:
            self.__ui.lblValue1.setText(self.__tr('Расстояние (м.):'))
            self.__ui.lblValue2.setText(self.__tr('Азимут (град.):'))

    def __set_items(self, cb, attrs, attrs_valid):
        cb.clear()
        cb.addItems([i for i in attrs if i is not None])
        for a in attrs_valid:
            if a and cb.findText(a.lower()) != -1:
                cb.setCurrentText(a)
                return
        cb.setCurrentIndex(-1)

    def __tablesIndexChanged(self, idx):
        t = self.__result_table()
        if t is not None:
            attrs = t.schema.attribute_names
            self.__set_items(self.__ui.cbGeometryId, attrs, [self.__last_used.field_geom, CoordinateProcess.IdFeatureField])
            self.__set_items(self.__ui.cbGeometryPartId, attrs, [self.__last_used.field_geom_part, CoordinateProcess.IdGeometryField])
            self.__ui.cbGeometryPartId.insertItem(0, self.NoneValue)
            self.__set_items(self.__ui.cbPointNumber, attrs, [self.__last_used.field_geom_point, CoordinateProcess.IdPointField])
            self.__ui.cbPointNumber.insertItem(0, self.NoneValue)
            self.__set_items(self.__ui.cbValue1, attrs, [self.__last_used.field_geom_value1] + CoordinateProcess.ValueFirstFieldList)
            self.__set_items(self.__ui.cbValue2, attrs, [self.__last_used.field_geom_value2] + CoordinateProcess.ValueSecondFieldList)

    def result_sql_text(self):
        # Результирующий SQL запрос
        def __ct( cb):
            return cb.currentText()
        sel = f'{__ct(self.__ui.cbGeometryId)}'
        order = f'int({__ct(self.__ui.cbGeometryId)})'
        if self.__ui.cbGeometryPartId.currentText() != self.NoneValue:
            p_id = __ct(self.__ui.cbGeometryPartId)
            sel = sel + f', {p_id}'
            order = order + f', abs(int({p_id}))'
        p_num_name = __ct(self.__ui.cbPointNumber)
        if p_num_name != self.NoneValue:
            t = self.__result_table()
            need_convert = False
            if t is not None:
                a = t.schema.by_name(p_num_name)
                if a:
                    need_convert = a.typedef in ['string']
            p_num = f', int({p_num_name})' if need_convert else f', {p_num_name}'
            sel = sel + p_num
            order = order + p_num
        sel = sel + f', {__ct(self.__ui.cbValue1)}, {__ct(self.__ui.cbValue2)}'
        return f'select {sel} from {self.__result_table().name} order by {order}'

    def result_coordsystem(self):
        return self._coordsystem

    def __result_table(self):
        # Исходная таблица
        return data_manager.find(self.__ui.cbTables.currentText())

    def result_type_coordinates(self):
        return TypeCoordinatesData(self.__ui.cbTypeData.currentIndex())


    def result_is_closed_polygon(self):
        return self.__ui.cbClosedAsPolygon.checkState() == Qt.Checked

    def result_type_action(self):
        return self.__ResultAction

    def result_filename(self):
        return self.__out_filename

    def result_has_part_number(self):
        return self.__ui.cbGeometryPartId.currentText() != self.NoneValue

    def result_has_point_number(self):
        return self.__ui.cbPointNumber.currentText() != self.NoneValue

    def exec(self):
        return self.__ui.exec()
        
