"""
Отразить по вертикали / горизонтали
"""

from PySide2.QtGui import QTransform
from axipy import tr, Text, Ellipse, Rectangle, RoundRectangle, Arc
from axipy.app import Notifications
from axipy.concurrent import task_manager, AxipyProgressHandler, ProgressSpecification, ProgressGuiFlags
from axipy.cs.CoordSystem import CoordSystem
from axipy.da import Feature, Table, data_manager
from axipy.da.Geometry import Point
from axipy.gui import view_manager
from axipy.interface import AxiomaInterface
from axipy.utl import Rect

from .. import helper

unsupported_types = (Text, Arc)
unsupported_types_string = ("Текст", "Дуга")

unsupported_single_geom_types = (Point, Ellipse, Rectangle, RoundRectangle)
unsupported_single_geom_types_string = ("Точка", "Эллипс", "Прямоугольник", "Скругленный прямоугольник")


class Mirror:
    """
    Базовый класс инструмента отражения геометрий.
    """

    def __init__(self, iface: AxiomaInterface, title: str) -> None:
        self.iface = iface
        self.title = title

    def transform_impl(self, bound: Rect) -> QTransform:
        pass

    def mirror(self, handler: AxipyProgressHandler, editable_table: Table, view_cs: CoordSystem) -> None:

        selection = data_manager.selection

        def is_allowed_feature(f: Feature) -> bool:
            return f.has_geometry() and not isinstance(f.geometry, unsupported_types)

        features = list(filter(lambda f: is_allowed_feature(f), selection.items()))

        n = len(features)
        if n == 0:
            self.notify_zero_features()
            return None
        elif n == 1:
            geom = features[0].geometry
            if geom.coordsystem == view_cs and isinstance(geom, unsupported_single_geom_types):
                self.notify_unsupported_single_geom()
                return None

        def reproject_feature_geom(feature):
            if feature.geometry.coordsystem != view_cs:
                feature.geometry = feature.geometry.reproject(view_cs)
            return feature

        features = [reproject_feature_geom(feature) for feature in features]

        # Получаем границы геометрии в координатах карты
        bounds = helper.quick_envelope([f.geometry for f in features])
        q_transform = self.transform_impl(bounds)

        handler.set_max_progress(n)
        for feature in features:
            if handler.is_canceled():
                return None
            geom = feature.geometry.affine_transform(q_transform)
            feature.geometry = geom
            handler.add_progress(1)
        handler.prepare_to_write_changes()
        editable_table.update(features)

        Notifications.push(self.title, tr("Было преобразовано геометрий: ") + str(n), Notifications.Success)

    def on_triggered(self) -> None:
        editable_table = helper.ensure_editable_table()
        view_cs = view_manager.active.coordsystem

        spec = ProgressSpecification(description=self.title, flags=ProgressGuiFlags.CANCELABLE)
        task_manager.run_and_get(spec, self.mirror, editable_table, view_cs)

    def notify_zero_features(self) -> None:
        Notifications.push(self.title, tr("Не удалось обновить ни одну геометрию. Список неподдерживаемых типов: ") +
                           ", ".join(unsupported_types_string), Notifications.Warning)

    def notify_unsupported_single_geom(self) -> None:
        Notifications.push(self.title, tr("Зеркальное отражение не поддерживается для одиночных геометрий типа: ") +
                           ", ".join(unsupported_single_geom_types_string), Notifications.Warning)


class HorizontalMirror(Mirror):
    """
    Отразить по горизонтали.
    """

    def __init__(self, iface: AxiomaInterface, title: str) -> None:
        super().__init__(iface, title)

    def transform_impl(self, bound: Rect) -> QTransform:
        dy = bound.ymin + bound.height / 2
        transform = QTransform().fromTranslate(0, 2 * dy)
        transform = transform.scale(1, -1)
        return transform


class VerticalMirror(Mirror):
    """
    Отразить по вертикали.
    """

    def __init__(self, iface: AxiomaInterface, title: str) -> None:
        super().__init__(iface, title)

    def transform_impl(self, bound: Rect) -> QTransform:
        dx = bound.xmin + bound.width / 2
        transform = QTransform().fromTranslate(2 * dx, 0)
        transform = transform.scale(-1, 1)
        return transform
