from typing import List
from PySide2.QtGui import QColor
from PySide2.QtCore import Qt
from axipy.cpp_render import (ShadowThematic, ShadowThematicRange,
                              ShadowThematicPie, ShadowThematicBar,
                              ShadowThematicSymbol, ShadowThematicIndividual,
                              ShadowThematicDensity)
from axipy.da import Style
from .layer import Layer, VectorLayer, ThematicLayer
from .context import Context
from typing import Any, Tuple


class StyledByIndexThematic:
    """Поддержка набора индексированных стилей.
    """

    def get_style(self, idx: int) -> Style:
        """Стиль для указанного выражения.

        Args:
            idx: Порядковый номер выражения.
        """
        return Style._wrap(self._get_style_impl(idx))
        

    def set_style(self, idx: int, style: Style):
        """Установка стиля оформления для выражения по его индексу в списке выражений.

        Args:
            idx: Индекс.
            style: Назначаемый стиль.

        .. literalinclude:: /../../tests/doc_examples/test_example_layer.py
            :caption: Пример установки стиля для значения с индексом 2 первого тематического слоя.
            :pyobject: test_run_example_layer_thematic
            :lines: 13-14
            :dedent: 4

        """
        self.shadow.set_style(idx, style.shadow)


class ReallocateThematicColor():
    """Поддержка различного рода алгоритмов распределения оформления."""

    def assign_two_colors(self, colorMin: QColor, colorMax: QColor, useHSV: bool = False):
        """Равномерно распределяет оформление по заданным крайним цветам.

        Args:
            colorMin: Цвет нижнего диапазона.
            colorMax: Цвет верхнего диапазона.
            useHSV: Если True, то будет использоваться схема HSV. В противном случае - RGB.
        """
        self.shadow.assign_two_colors(colorMin, colorMax, useHSV)

    def assign_three_colors(self, colorMin: QColor, colorMax: QColor, colorBreak: QColor, br: int, useHSV: bool = True):
        """Цвет, распределенный между тремя заданными цветами (с разрывом).
        
        Args:
            colorMin: Цвет нижнего диапазона.
            colorMax: Цвет верхнего диапазона.
            colorBreak: Цвет на уровне разрыва.
            br: Индекс интервала, на на котором используется цвет разрыва.
            useHSV: Если True, то будет использоваться схема HSV. В противном случае - RGB.
        """
        self.shadow.assign_three_colors(colorMin, colorMax, colorBreak, br, useHSV)

    def assign_rainbow(self, sequential: bool = True, saturation: float = 90, value: float = 90):
        """Распределение цветов по спектру. Цветовая схема HSV.
        
        Args:
            sequential: Если True, то последовательное распределение цветов. В противном случае распределение случайно.
            saturation: Яркость. Задается в интервале (0..100)
            value: Насыщенность. Задается в интервале (0..100)
        """
        self.shadow.assign_rainbow(sequential, saturation / 100, value / 100)

    def assign_gray(self, minV: int = 20, maxV: int = 80):
        """Распределение в виде градации серого. Значение задается в интервале (0..100) от черного до белого.

        Args:
            minV: Минимальное значение.
            maxV: Максимальное значение.
        """
        self.shadow.assign_gray(minV, maxV)

    def assign_monotone(self, color: QColor, minv: int = 20, maxv: int = 80):
        """Монотонная заливка разной яркости (оттенки красного, синего и т.п.). Цветовая схема HSL.
        Максимальное и минимальное значения задаются в интервале (0..100).

        Args:
            color: Базовый цвет.
            minV: Минимальное значение.
            maxV: Максимальное значение.
        """
        self.shadow.assign_monotone(color, minv, maxv)

class RangeThematicLayer(ThematicLayer, StyledByIndexThematic, ReallocateThematicColor):
    """Тематическое оформление слоя с распределением значений по интервалам. Для распределения цветов по
    заданным интервалам могут быть использованы функции `assign_*` класса :class:`ReallocateThematicColor`
    в зависимости от требуемых целей.

    Args:
        expression: Наименование атрибута таблицы или выражение.


    .. literalinclude:: /../../tests/doc_examples/test_example_layer.py
        :caption: Пример создания тематики по интервалам.
        :pyobject: test_run_example_layer_thematic_range
        :lines: 3-
        :dedent: 4
    """

    def __init__(self, expression: str):
        self.shadow = ShadowThematicRange(expression)

    @classmethod
    def _wrap(cls, _shadow: ShadowThematicRange):
        obj = cls.__new__(cls)
        obj.shadow = _shadow
        return obj

    @property
    def ranges(self) -> int:
        """Количество интервалов."""
        return self.shadow.get_ranges()

    @ranges.setter
    def ranges(self, n: int):
        self.shadow.set_ranges(n)

    EQUAL_INTERVAL = 1
    EQUAL_COUNT = 2
    MANUAL = 3

    @property
    def splitType(self) -> int:
        """Тип распределения значений по интервалам.

        .. csv-table:: Допустимые значения:
            :header: "Константа", "Значение", "Описание"
            :align: left
            :widths: 10, 5, 40

            EQUAL_INTERVAL, 1, "Распределение исходя из равномерности интервалов (по умолчанию)."
            EQUAL_COUNT, 2, "Распределение исходя их равного количества объектов в каждом интервале."
            MANUAL, 3, "Ручное распределение значений путем задания пределов вручную."
        """
        return self.shadow.get_split_type()

    @splitType.setter
    def splitType(self, n: int):
        self.shadow.set_split_type(n)

    def get_interval_value(self, idx: int) -> Tuple[float, float]:
        """Возвращает предельные значения для указанного интервала в виде пары значений.

        Args:
            idx: Индекс диапазона.
        """
        return self.shadow.get_interval_values(idx)

    def set_interval_value(self, idx: int, v: Tuple[float, float]):
        """Заменяет предельные значения интервала.

        Args:
            idx: Индекс диапазона.
            v: Значение в виде пары.

        """
        self.shadow.set_interval_values(idx, v)

    def _get_style_impl(self, idx):
        return ShadowThematicRange.get_style(self.shadow, idx)

def create_range(expression: str, ranges: int = 6, colorMin=Qt.yellow, colorMax=Qt.red, splitType=RangeThematicLayer.EQUAL_INTERVAL):
    range = RangeThematicLayer(expression)
    range.ranges = ranges
    range.assign_two_colors(colorMin, colorMax)
    range.splitType = splitType
    return range


class AllocationThematic:
    """Метод распределения значений для диаграмм."""

    LINEAR = 1
    SQRT = 2
    LOG10 = 3

    @property
    def allocationType(self) -> int:
        """Тип распределения значений.

        .. csv-table:: Допустимые значения.
            :header: "Константа", "Значение", "Описание"
            :align: left
            :widths: 10, 10, 30

            LINEAR, 1, "Линейное (по умолчанию)"
            SQRT, 2, "Квадратичное"
            LOG10, 3, "Логарифмическое"
        """
        return self.shadow.get_allocation_type()

    @allocationType.setter
    def allocationType(self, n: int):
        self.shadow.set_allocation_type(n)


class OrientationThematic:
    """Ориентация тематического представления относительно центроида объекта."""

    CENTER = 0,
    LEFT_UP = 1
    UP = 2
    RIGHT_UP = 3
    RIGHT = 4
    RIGHT_DOWN = 5
    DOWN = 6
    LEFT_DOWN = 7
    LEFT = 8

    @property
    def orientationType(self) -> int:
        """Ориентация относительно центроида.

        .. csv-table:: Допустимые значения.
            :header: "Константа", "Значение", "Описание"
            :align: left
            :widths: 10, 10, 30

            CENTER, 0, "Диаграмма рисуется по центру (по умолчанию)"
            LEFT_UP, 1, "Диаграмма выравнивается по левому верхнему краю"
            UP, 2, "Диаграмма выравнивается по верхнему краю"
            RIGHT_UP, 3, "Диаграмма выравнивается по верхнему правому краю"
            RIGHT, 4, "Диаграмма выравнивается по правому краю"
            RIGHT_DOWN, 5, "Диаграмма выравнивается по нижнему правому краю"
            DOWN, 6, "Диаграмма выравнивается по нижнему краю"
            LEFT_DOWN, 7, "Диаграмма выравнивается по нижнему левому краю"
            LEFT, 8, "Диаграмма выравнивается по левому краю"
        """
        return self.shadow.get_orientation_type()

    @orientationType.setter
    def orientationType(self, n: int):
        self.shadow.set_orientation_type(n)


class PieThematicLayer(ThematicLayer, AllocationThematic, OrientationThematic, StyledByIndexThematic):
    """Тематика в виде круговых диаграмм.

    Args:
        expressions: Наименования атрибутов или выражений в виде списка :class:`list`.

    .. literalinclude:: /../../tests/doc_examples/test_example_layer.py
        :caption: Создание тематики с последующим добавлением ее к базовому слою.
        :pyobject: test_run_example_layer_thematic_pie
        :lines: 3-
        :dedent: 4
    """

    def __init__(self, expressions: List):
        self.shadow = ShadowThematicPie(expressions)

    @classmethod
    def _wrap(cls, _shadow: ShadowThematicPie):
        obj = cls.__new__(cls)
        obj.shadow = _shadow
        return obj

    @property
    def startAngle(self) -> bool:
        """Начальный угол отсчета диаграммы.
        """
        return self.shadow.get_start_angle()

    @startAngle.setter
    def startAngle(self, v: float):
        self.shadow.set_start_angled(v)

    def _get_style_impl(self, idx):
        return ShadowThematicPie.get_style(self.shadow, idx)



def create_pie(expressions: List, allocationType: int = AllocationThematic.LINEAR):
    pie = PieThematicLayer(expressions)
    pie.allocationType = allocationType
    return pie


class BarThematicLayer(ThematicLayer, AllocationThematic, OrientationThematic, StyledByIndexThematic):
    """Тематика в виде столбчатых диаграмм.

    Args:
        expressions: Наименования атрибутов или выражений в виде списка :class:`list`.

    .. literalinclude:: /../../tests/doc_examples/test_example_layer.py
        :caption: Создание тематики с последующим добавлением ее к базовому слою.
        :pyobject: test_run_example_layer_thematic_bar
        :lines: 3-
        :dedent: 4
    """

    def __init__(self, expressions: List):
        self.shadow = ShadowThematicBar(expressions)

    @classmethod
    def _wrap(cls, _shadow: ShadowThematicBar):
        obj = cls.__new__(cls)
        obj.shadow = _shadow
        return obj

    @property
    def isStacked(self) -> bool:
        """Расположение столбчатой диаграммы в виде стопки, если True.
        """
        return self.shadow.get_is_stacked()

    @isStacked.setter
    def isStacked(self, v: bool):
        self.shadow.set_is_stacked(v)

    def _get_style_impl(self, idx):
        return ShadowThematicBar.get_style(self.shadow, idx)


class SymbolThematicLayer(ThematicLayer):
    """Тематический слой с распределением по интервалам и с градуировкой символа по размеру.

    Args:
        expression: Наименование атрибута или выражение.

    .. literalinclude:: /../../tests/doc_examples/test_example_layer.py
        :caption: Создание тематики с последующим добавлением ее к базовому слою.
        :pyobject: test_run_example_layer_thematic_symbol
        :lines: 3-
        :dedent: 4
    """

    def __init__(self, expression: str):
        self.shadow = ShadowThematicSymbol(expression)

    @classmethod
    def _wrap(cls, _shadow: ShadowThematicSymbol):
        obj = cls.__new__(cls)
        obj.shadow = _shadow
        return obj

    @property
    def minHeight(self) -> float:
        """Минимальная высота символа.
        """
        return self.shadow.get_min_height()

    @minHeight.setter
    def minHeight(self, v: float):
        self.shadow.set_min_height(v)

    @property
    def maxHeight(self) -> float:
        """Максимальная высота символа.
        """
        return self.shadow.get_max_height()

    @maxHeight.setter
    def maxHeight(self, v: float):
        self.shadow.set_max_height(v)

    @property
    def defaultStyle(self) -> Style:
        """Стиль по умолчанию для оформления знаков.
        """
        return Style._wrap(self.shadow.get_default_style())

    @defaultStyle.setter
    def defaultStyle(self, style: Style):
        self.shadow.set_default_style(style.shadow)


class IndividualThematicLayer(ThematicLayer, StyledByIndexThematic, ReallocateThematicColor):
    """Тематический слой с распределением стилей по индивидуальным значением.

    Args:
        expression: Наименование атрибута или выражение.

    .. literalinclude:: /../../tests/doc_examples/test_example_layer.py
        :caption: Создание тематики с последующим добавлением ее к базовому слою.
        :pyobject: test_run_example_layer_thematic_individual
        :lines: 3-
        :dedent: 4
    """

    def __init__(self, expression: str):
        self.shadow = ShadowThematicIndividual(expression)

    @classmethod
    def _wrap(cls, _shadow: ShadowThematicIndividual):
        obj = cls.__new__(cls)
        obj.shadow = _shadow
        return obj

    @property
    def count(self):
        """Количество значений в тематике."""
        return self.shadow.get_count()

    def get_value(self, idx: int) -> Any:
        """Выражение по указанному индексу.

        Args:
            idx: Индекс.
        """
        return self.shadow.get_value(idx)

    def _get_style_impl(self, idx):
        return ShadowThematicIndividual.get_style(self.shadow, idx)

    def __len__(self):
        return self.count


class DensityThematicLayer(ThematicLayer):
    """Тематический слой с заполнением полигональных объектов точками, плотность которых зависит от вычисленного значения по выражению.

    Args:
        expression: Наименование атрибута или выражение.

    .. literalinclude:: /../../tests/doc_examples/test_example_layer.py
        :caption: Создание тематики с последующим добавлением ее к базовому слою.
        :pyobject: test_run_example_layer_thematic_density
        :lines: 3-
        :dedent: 4
    """

    def __init__(self, expression: str):
        self.shadow = ShadowThematicDensity(expression)

    @classmethod
    def _wrap(cls, _shadow: ShadowThematicDensity):
        obj = cls.__new__(cls)
        obj.shadow = _shadow
        return obj

    @property
    def pointForMaximum(self) -> int:
        """Количество точек для максимального значения."""
        return self.shadow.get_point_for_maximum()

    @pointForMaximum.setter
    def pointForMaximum(self, cnt: int):
        return self.shadow.set_point_for_maximum(cnt)

    @property
    def color(self) -> QColor:
        """Цвет точек."""
        return self.shadow.get_color()

    @color.setter
    def color(self, c: QColor):
        return self.shadow.set_color(c)

    @property
    def size(self) -> float:
        """Размер точек."""
        return self.shadow.get_size()

    @size.setter
    def size(self, v: float):
        return self.shadow.set_size(v)
