from .data_provider import DataProvider, Table
from .source import Source, Destination, Schema
from typing import List, Optional


class MsSqlSource(Source):
    pass


class MsSqlDataProvider(DataProvider):
    """Провайдер для Базы Данных MSSQLServer.
    
    Note:
        Для работы с СУБД `Microsoft SQL Server` необходимо скачать и установить `Microsoft
        SQL Server Native Client`.
        
        См. Руководство по установке и активации.

    Note:
        Ссылку на провайдер можно получить через глобальную переменную :attr:`axipy.da.provider_manager.mssql`.
    """
    _identifier = 'MsSqlDataProvider'

    def get_source(self, host: str, db_name: str, user: str,
                   password: str, port: int = 1433, dataobject: Optional[str] = None, sql : Optional[str] = None, alias: str = None) -> Source:
        """Создает описательную структуру для источника данных. Она в дальнейшем может быть использована при открытии данных :meth:`ProviderManager.open`.

        Note:
            В качестве таблицы можно указать либо ее наименование `dataobject` либо текст запроса `sql`.

        Note:
            Ссылку на провайдер можно получить через глобальную переменную :attr:`axipy.provider_manager.mssql`.

        Args:
            host: Адрес сервера.
            db_name: Имя базы данных.
            user: Имя пользователя.
            password: Пароль.
            port: Порт.
            dataobject: Имя таблицы.
            sql: SQL-запрос. Если указан, то он имеет более высокий приоритет по отношению к значению `dataobject`.

        Пример с указанием имени таблицы::

            definition = provider_manager.mssql.get_source('localhost', 'test', 'sa', 'sa', dataobject='world')
            table = provider_manager.open(definition)

        Пример с указанием текста запроса::

            definition = provider_manager.mssql.get_source('localhost', 'test', 'sa', 'sa', sql="select * from world where Страна like 'Р%'")
            table = provider_manager.open(definition)
        """
        return MsSqlSource(
            Source._provider(self.id),
            Source._alias(alias),
            {
                'host': host,
                'port': port,
                'db': db_name,
                'user': user,
                'password': password,
                'dataobject': dataobject,
                'sql' : sql
            }
        )

    def open(self, host: str, db_name: str, user: str,
                   password: str, port: int = 1433, dataobject: Optional[str] = None, sql: Optional[str] = None, alias: str = None) -> Table:
        """Открывает объект данных.

        Note:
            В качестве таблицы можно указать либо ее наименование `dataobject` либо текст запроса `sql`.

        Args:
            host: Адрес сервера.
            db_name: Имя базы данных.
            user: Имя пользователя.
            password: Пароль.
            port: Порт.
            dataobject: Имя таблицы.
            sql: SQL-запрос. Если указан, то он имеет более высокий приоритет по отношению к значению `dataobject`.
            alias: Псевдоним для открываемой таблицы.
        """
        return self.get_source(host, db_name, user, password, port, dataobject, sql, alias).open()
