from typing import List

import axipy
from PySide2.QtCore import QObject, Signal
from axipy.app import Notifications
from axipy.concurrent import task_manager, AxipyProgressHandler, ProgressGuiFlags, ProgressSpecification
from axipy.da import (
    Geometry,
    Style,
    CollectionStyle,
    PointStyle,
    LineStyle,
    PolygonStyle,
    TextStyle,
    Table,
    Feature,
    GeometryCollection,
    ValueObserver,
    state_manager,
    DefaultKeys)
from axipy.da import data_manager as dm
from axipy.interface import AxiomaInterface

from ..helper import ensure_editable_table

paste_style_observer_key = "PasteStyleObserver"


def flat_collection_style(collection_style: CollectionStyle) -> List[Style]:
    styles = [collection_style.point(), collection_style.line(),
              collection_style.polygon(), collection_style.text()]
    return list(filter(None, styles))


def try_paste_to_collection_style_impl(
        style: Style, collection_style: CollectionStyle) -> bool:
    if isinstance(style, PolygonStyle):
        if collection_style.polygon() is not None and collection_style.polygon() != style:
            return False
        collection_style.for_polygon(style)
    elif isinstance(style, LineStyle):
        if collection_style.line() is not None and collection_style.line() != style:
            return False
        collection_style.for_line(style)
    elif isinstance(style, PointStyle):
        if collection_style.point() is not None and collection_style.point() != style:
            return False
        collection_style.for_point(style)
    elif isinstance(style, TextStyle):
        if collection_style.text() is not None and collection_style.text() != style:
            return False
        collection_style.for_text(style)
    return True


def try_paste_to_collection_style(
        style: Style, collection_style: CollectionStyle) -> bool:
    # Если в хранилище уже есть стиль для этого типа геометрии и он отличается от
    # нового, значит в выборке пристутствуют несколько стилей для одного типа
    # геометрий. Что приведёт к неочивидном результату. Скопируется только один из
    # них причем никто не гарантирует какой именно.
    if isinstance(style, CollectionStyle):
        for style in flat_collection_style(style):
            success = try_paste_to_collection_style_impl(
                style, collection_style)
            if not success:
                return False
        return True
    else:
        return try_paste_to_collection_style_impl(style, collection_style)


def merge_collection(source: CollectionStyle, target: CollectionStyle):
    """
    Копируем только те стили из коллекции, которые там есть
    """
    if source.polygon() is not None:
        target.for_polygon(source.polygon())
    if source.line() is not None:
        target.for_line(source.line())
    if source.point() is not None:
        target.for_point(source.point())
    if source.text() is not None:
        target.for_text(source.text())
    return target


class StyleStorage(QObject):
    """
    Буфер для хранения стилей при операциях Копирования/Вставки стилей
    """
    state_changed = Signal(bool)

    def __init__(self) -> None:
        super().__init__(None)
        self.__collection_style = self.__default_collection_style()  # type: CollectionStyle

    def for_geometry(self, geom: Geometry, style: Style) -> Style:
        # Для коллекций разнородных объектов
        if issubclass(type(geom), GeometryCollection) \
                and issubclass(type(style), CollectionStyle):
            return merge_collection(self.__collection_style, style)
        # Для коллекций однородных геометрий
        if issubclass(type(geom), GeometryCollection):
            return self.__get_style_for_single_geometry_collection(style)
        # Для обычных одиночных геометрий
        return self.__collection_style.find_style(geom)

    def __get_style_for_single_geometry_collection(self, style) -> Style:
        # Если у геометрии нет стиля
        if style is None:
            return None
        if isinstance(
                style,
                PolygonStyle) and self.__collection_style.polygon() is not None:
            return self.__collection_style.polygon()
        if isinstance(
                style,
                PointStyle) and self.__collection_style.point() is not None:
            return self.__collection_style.point()
        if isinstance(
                style,
                LineStyle) and self.__collection_style.line() is not None:
            return self.__collection_style.line()
        if isinstance(
                style,
                TextStyle) and self.__collection_style.text() is not None:
            return self.__collection_style.text()
        return None

    def set_style(self, style: Style) -> None:
        if isinstance(style, PolygonStyle):
            self.__collection_style.for_polygon(style)
        elif isinstance(style, LineStyle):
            self.__collection_style.for_line(style)
        elif isinstance(style, PointStyle):
            self.__collection_style.for_point(style)
        elif isinstance(style, TextStyle):
            self.__collection_style.for_text(style)
        elif isinstance(style, CollectionStyle):
            merge_collection(style, self.__collection_style)
        else:
            print(f"Неподдерживаемый тип стиля {style}")
        self.state_changed.emit(self.is_empty())

    def is_empty(self) -> bool:
        return self.__collection_style.text() is None and self.__collection_style.point() is None \
            and self.__collection_style.polygon() is None and self.__collection_style.line() is None

    def __default_collection_style(self):
        return CollectionStyle()

    def clean(self):
        self.__collection_style = self.__default_collection_style()
        self.state_changed.emit(self.is_empty())


def register_paste_style_observer(storage: StyleStorage) -> ValueObserver:
    """
    Создаём наблюдателя чтобы управлять доступностью кнопки вставить стиль.
    Её доступность отличается от Копировать тем, что для вставки необходим
    заполненый буфер стилей
    """
    editable_observer = state_manager.find(DefaultKeys.SelectionEditableIsSame)

    def is_enabled():
        return not storage.is_empty() and editable_observer.value()

    paste_style_observer = state_manager.find(paste_style_observer_key)
    if paste_style_observer is None:
        paste_style_observer = state_manager.create(
            paste_style_observer_key, is_enabled())

    def set_value():
        paste_style_observer.setValue(is_enabled())

    editable_observer.changed.connect(set_value)
    storage.state_changed.connect(set_value)
    # выставляем начальное значение
    set_value()
    return paste_style_observer


class CopyStyle:
    """
    Копируем стиль выделенных объектов в буфер
    """

    def __init__(
            self,
            iface: AxiomaInterface,
            title: str,
            storage: StyleStorage) -> None:
        self.title = title
        self.iface = iface
        self.__storage = storage

    def __notify_about_the_same_style(self):
        message = axipy.tr("В выборке содержится несколько"
                           " стилей для одного и того же типа геометрии.")
        self.iface.notifications.push(
            self.title, message, Notifications.Critical)

    def __notify_about_copy_styles(self, cout: int):
        if cout == 0:
            message = axipy.tr(
                "Не получилось скопировать ни один стиль. "
                "Копирование стиля для текстовых объектов не поддерживается.")
            self.iface.notifications.push(
                self.title, message, Notifications.Critical)
        else:
            message = axipy.tr(f"Было скопированно стилей: {cout}")
            self.iface.notifications.push(
                self.title, message, Notifications.Success)

    def __copy_style(self, handler: AxipyProgressHandler):
        self.__storage.clean()
        selection_table = dm.selection  # type: Table
        handler.set_max_progress(selection_table.count())
        found_styles = CollectionStyle()
        for feature in selection_table:
            handler.raise_if_canceled()
            if not feature.has_style():
                continue
            success = try_paste_to_collection_style(
                feature.style, found_styles)
            if not success:
                self.__notify_about_the_same_style()
                return
            handler.add_progress(1)
        self.__storage.set_style(found_styles)
        self.__notify_about_copy_styles(
            len(flat_collection_style(found_styles)))

    def on_triggered(self):
        spec = ProgressSpecification(
            description=self.title,
            flags=ProgressGuiFlags.CANCELABLE)
        task_manager.run_and_get(spec, self.__copy_style)


class PasteStyle:
    """
    Вставляем стиль из буфера в выделенные объекты
    """

    def __init__(
            self,
            iface: AxiomaInterface,
            title: str,
            storage: StyleStorage) -> None:
        self.title = title
        self.iface = iface
        self.__storage = storage

    def __success_notification(self, amout_of_features: int):
        if amout_of_features == 0:
            self.iface.notifications.push(
                self.title,
                axipy.tr(
                    "Ни у одного объекта не удалось сменить стиль."
                    " Попробуйте скопировать другой стиль перед тем как применить операцию вставить."),
                Notifications.Warning)
            return
        self.iface.notifications.push(
            self.title, axipy.tr(
                f"Количество объектов, у которых был изменён стиль: {amout_of_features}"),
            Notifications.Success)

    def __paste_operation(self, handler: AxipyProgressHandler):
        selection_table = dm.selection  # type: Table
        features = []  # type: List[Feature]
        handler.set_max_progress(selection_table.count)
        for feature in selection_table:
            handler.raise_if_canceled()
            if not feature.has_geometry() or not feature.has_style():
                continue
            style = self.__storage.for_geometry(
                feature.geometry, feature.style)
            if style is None:
                continue
            feature.style = style
            features.append(feature)
            handler.add_progress(1)
        table = ensure_editable_table()
        handler.prepare_to_write_changes()
        table.update(features)
        self.__success_notification(len(features))

    def on_triggered(self):
        spec = ProgressSpecification(
            description=self.title,
            flags=ProgressGuiFlags.CANCELABLE)
        task_manager.run_and_get(spec, self.__paste_operation)
