from typing import List, Union, Optional
from PySide2.QtCore import QObject, Signal
from PySide2.QtGui import QImage
from axipy.cs import CoordSystem, LinearUnit, AreaUnit
from axipy.utl import Rect
from axipy.cpp_render import ShadowMap, ShadowVectorLayer, ShadowRasterLayer, ShadowLayers, ShadowLayer
from .context import Context
from .layer import Layer, VectorLayer, RasterLayer, CosmeticLayer


class ListLayers:
    """Группа слоев. Может включать в себя как слои :class:`axipy.render.Layer` так и
    группы слоев :class:`axipy.render.ListLayers`.
    Пример использования см :attr:`axipy.render.Map.layers`
    """

    def __init__(self):
        raise NotImplementedError

    @classmethod
    def _wrap(cls, shadow):
        result = cls.__new__(cls)
        result.shadow = shadow
        return result

    @property
    def count(self) -> int:
        """Количество слоев и групп слоев. Так же допустимо использование функции :meth:`len`
        """
        return self.__len__()

    def __len__(self):
       return self.shadow.count()
    
    def __iter__(self):
       return (self.__getitem__(idx) for idx in range(self.__len__()))

    def at(self, index: int) -> Union[Layer, 'ListLayers']:
        """Возвращает слой или группы слоев по их индексу.

        Args:
          index: Индекс слоя или группы в списке.

        Например::

            layers.at(2)
            layers[2]
        """
        return self.__getitem__(index)

    def __getlayer(self, item):
        if isinstance(item, ShadowLayer):
            return Layer._wrap(item)
        elif isinstance(item, ShadowLayers):
            return ListLayers._wrap(item)
        return item

    def __getitem__(self, index: Union[int, str]) -> Union[Layer, 'ListLayers']:
        """Возвращает слой или группу по их индексу.

        Args:
            index: Индекс слоя или группы слоев.

        Returns:
            Искомый элемент.
        """
        if isinstance(index, str):
            key = index
            result = ShadowLayers.by_name(self.shadow, key)
            if result is None:
                raise KeyError(f'Layer "{key}" is not found.')
            return self.__getlayer(result)
        if index < 0 or index >= self.count:
            raise IndexError(f'Index "{index}" is out of range.')
        item = ShadowLayers.at(self.shadow, index)
        return self.__getlayer(item)

    @property
    def title(self) -> str:
        """Наименование группы."""
        return self.shadow.get_name()

    @title.setter
    def title(self, n: str):
        self.shadow.set_name(n)

    def append(self, layer: Layer):
        """Добавляет слой в карту. Добавление группы слоев не поддерживается и производится
        путем группировки существующих элементов посредством метода :meth:`group`.

        Args:
          layer: Добавляемый слой.
        
        Raises:
            ValueError: Если слой уже содержится в карте.
        """
        if layer in self:
            raise ValueError('Слой уже содержится в карте.')
        self.shadow.add(layer.shadow)

    def add(self, layer: Layer):
        self.append(layer)

    def remove(self, index: int):
        """Удаляет слой по индексу.

        Args:
          index: Индекс удаляемого слоя.
        """
        self.shadow.remove(index)

    def move(self, from_index: int, to_index: int):
        """Перемещает слой или вложенную группу слоев в списке слоев по его индексу.

        Args:
          from_index: Индекс слоя для перемещения.
          to_index: Целевой индекс.
        """
        self.shadow.move(from_index, to_index)

    def group(self, indexes: List[int], name: str):
        """Группировка слоев и групп в соответствие со списком их индексов.
        При этом создается новая группа и все элементы (слои и группы слоев) помещаются
        внутрь этой группы.
        
        Args:
            indexes: Список индексов элементов, которые необходимо объединить.
            name: Наименование создаваемой группы.
        """
        self.shadow.group(indexes, name)

    def ungroup(self, index: int):
        """Разгруппировка группы слоев по его индексу. при этом все внутренние элементы
        переносятся на верхний уровень данного списка. Если по индексу располагается не
        группа, то будет выброшено исключение.
        
        Args:
            index: Индекс группы слоев.
        """
        self.shadow.ungroup(index)

    def add_group(self, name: str):
        """Создает пустую группу.
        
        Args:
            name: Наименование создаваемой группы.
        """
        self.shadow.add_group(name)

class Map:
    """Класс карты. Рассматривается как группа слоев, объединенная в единую сущность.
    Вне зависимости от СК входящих в карту слоев, карта отображает все слои в одной СК.
    Найти наиболее подходящую для этого можно с помощью :meth:`get_best_coordsystem` или
    же установить другую.

    Единицы измерения расстояний :attr:`distanceUnit` и площадей :attr:`areaUnit` берутся из настроек по умолчанию.

    Args:
        layers: Список слоев, с которым будет создана карта.
    
    Raises:
        ValueError: Если один и тот же слой был передан несколько раз.

    .. literalinclude:: /../../tests/doc_examples/test_example_map.py
        :caption: Пример.
        :pyobject: test_run_example_map
        :lines: 3-
        :dedent: 4
    """

    def __init__(self, layers: List[Layer] = []):
        from axipy.app import render_instance
        super().__init__()
        to_layer = lambda l: l if isinstance(l, Layer) else Layer.create(l)
        layers = list(map(to_layer, layers))
        if any(layers.count(l) > 1 for l in layers):
            raise ValueError('Один слой был добавлен несколько раз.')
        get_shadow = lambda obj: obj.shadow
        shadowlayers = map(get_shadow, layers)
        self.shadow = ShadowMap(list(shadowlayers), render_instance)
        self._layers = None

    @classmethod
    def _wrap(cls, shadow: ShadowMap):
        result = cls.__new__(cls)
        result.shadow = shadow
        result._layers = None
        return result

    @property
    def need_redraw(self) -> Signal:
        """``Signal[]`` Сигнал о необходимости перерисовки карты. 
        Возникает при изменении контента одного или нескольких слоев карты. 
        Это может быть обусловлено изменением данных таблиц.

        .. literalinclude:: /../../tests/doc_examples/test_example_map.py
            :caption: Пример.
            :pyobject: test_run_example_map_redraw
            :lines: 2-
            :dedent: 4
        """
        return self.shadow.needRedraw

    def draw(self, context: Context):
        """Рисует карту в контексте.

        Args:
            context: Контекст рисования.

        .. literalinclude:: /../../tests/doc_examples/test_example_map.py
            :caption: Пример.
            :pyobject: test_run_example_map_draw
            :lines: 3-
            :dedent: 4
        """
        self.shadow.draw(context.shadow)

    def get_best_coordsystem(self) -> CoordSystem:
        """Определяет координатную системы карты, наиболее подходящую
        исходя из содержимого перечня слоев.
        """
        return CoordSystem._wrap(self.shadow.bestCS())

    def get_best_rect(self, coordsystem: CoordSystem = None) -> Rect:
        """Определяет ограничивающий прямоугольник карты.

        Args:
            coordsystem:
                Координатная система, в которой необходимо получить результат.
                Если отсутствует, будет выдан результат для наиболее подходящей
                координатной системы.
        """
        if coordsystem is None:
            coordsystem = self.get_best_coordsystem()
        return Rect.from_qt(self.shadow.bestRect(coordsystem.shadow))

    @property
    def distanceUnit(self) -> LinearUnit:
        """Единицы измерения расстояний на карте."""
        return LinearUnit._wrap(self.shadow.get_distance_unit())

    @distanceUnit.setter
    def distanceUnit(self, unit: LinearUnit):
        self.shadow.set_distance_unit(unit.shadow)

    @property
    def areaUnit(self) -> AreaUnit:
        """Единицы измерения площадей карты."""
        return AreaUnit._wrap(self.shadow.get_area_unit())

    @areaUnit.setter
    def areaUnit(self, u: AreaUnit):
        self.shadow.set_area_unit(u.shadow)

    @property
    def layers(self) -> ListLayers:
        """Список слоев и групп слоев.

        Note:
            Не содержит косметический слой :attr:`cosmetic`.

        .. literalinclude:: /../../tests/doc_examples/test_example_map.py
            :caption: Примеры доступа.
            :pyobject: test_run_example_map_layers
            :lines: 5-
            :dedent: 4
        """
        if self._layers is None:
            self._layers = ListLayers._wrap(ShadowMap.list_layers(self.shadow))
        return self._layers


    @property
    def editable_layer(self) -> VectorLayer:
        """Слой, установленный для текущего редактирования в карте.
        
        Raises:
            ValueError: При попытке установить слой, не принадлежащий этой
                        карте.
        """
        el = ShadowMap.get_editableLayer(self.shadow)
        return Layer._wrap(el)

    @editable_layer.setter
    def editable_layer(self, layer: Optional[VectorLayer]):
        if layer is None:
            return self.shadow.reset_editableLayer()
        if not layer in self:
            raise ValueError('Слой не принадлежит карте.')
        self.shadow.set_editableLayer(layer.shadow)

    @property
    def cosmetic(self) -> CosmeticLayer:
        """Косметический слой карты."""
        return Layer._wrap( ShadowMap.cosmetic(self.shadow))

    def __eq__(self, other):
        if not isinstance(other, Map):
            return NotImplementedError
        return other is not None and self.shadow.isEqual(other.shadow)

    def __str__(self):
        return str(type(self))

    def __contains__(self, layer: Layer):
        return layer in self.layers or layer == self.cosmetic

    def to_image(self, width: int, height: int, coordsystem: CoordSystem = None, bbox: Rect = None) -> QImage:
        """Рисует карту в изображение.

        Args:
            width: Ширина выходного изображения.
            height: Высота выходного изображения.
            coordsystem: Координатная система. Если не задана, берется наиболее подходящая.
            bbox: Ограничивающий прямоугольник. Если не задан, берется у карты.

        Returns: 
            Изображение.
        """
        from PySide2.QtGui import QPainter
        image = QImage(width, height, QImage.Format_ARGB32_Premultiplied)
        image.fill(0)
        painter = QPainter(image)
        context = Context(painter)
        if coordsystem is None:
            coordsystem = self.get_best_coordsystem()
        if bbox is None:
            bbox = self.get_best_rect(coordsystem)
        if isinstance(bbox, tuple):
            bbox = Rect(*bbox)
        context.coordsystem = coordsystem
        context.rect = bbox
        self.draw(context)
        return image
