import sys
import os.path
import importlib
from typing import Union
from axipy.axparser import *


def _remove_plugin(name: str):
    if name not in sys.modules:
        return
    del sys.modules[name]
    def is_submodule(subname): return subname.startswith(name + '.')
    submodules = list(filter(is_submodule, sys.modules.keys()))
    for submodule in submodules:
        del sys.modules[submodule]


def load_plugin_dir(path: str, name: str):
    _remove_plugin(name)
    spec = importlib.util.spec_from_file_location(
        name, os.path.join(path, name, '__init__.py'))
    module = importlib.util.module_from_spec(spec)
    sys.modules[name] = module
    spec.loader.exec_module(module)
    return module


def init_translation(path, filename_base):
    from PySide2.QtCore import QTranslator, QCoreApplication
    import axipy.app
    translator = QTranslator()
    status = translator.load(
        axipy.app.core_instance.translationLocale(), filename_base, "_", path)
    if not status:
        return
    if QCoreApplication.installTranslator(translator):
        translator.setParent(QCoreApplication.instance())


def loadPlugin(absolute_path: str):
    from axipy.axioma_plugin import AxiomaPlugin
    path, name = os.path.split(absolute_path)
    plugin_dir = os.path.join(path, name)
    init_translation(os.path.join(plugin_dir, 'i18n'), 'translation')
    module = load_plugin_dir(path, name)
    if not hasattr(module, 'Plugin'):
        return None
    plugin_decl = getattr(module, 'Plugin')
    if issubclass(plugin_decl, AxiomaPlugin):
        return plugin_decl._axioma_plugin_create(plugin_dir)
    import inspect
    signature = inspect.signature(plugin_decl.__init__)
    if 'iface' not in signature.parameters:
        return plugin_decl()
    from axipy.interface import AxiomaInterface
    axioma_interface = AxiomaInterface()
    axioma_interface.initialize(plugin_dir)
    return plugin_decl(axioma_interface)
