from axipy.cpp_core_dp import ShadowDataCatalog
from PySide2.QtCore import QObject, Signal
from .DataObjectWrapper import DataObject, Table
from typing import Union, Optional, List, Callable


class DataManager:
    """Хранилище объектов данных.
    При открытии таблицы или растра эти объекты автоматически попадают в данный каталог.
    Для отслеживания изменений в каталоге используются события :attr:`added` и :attr:`removed`.

    Note:
        Используйте готовый экземпляр этого класса
        :attr:`axipy.da.data_manager`.

    .. literalinclude:: /../../tests/doc_examples/test_example_dp.py
        :caption: Пример использования.
        :pyobject: test_run_example_catalog
        :lines: 3-
        :dedent: 4
    """

    def __init__(self):
        self.shadow = ShadowDataCatalog()

    @classmethod
    def _wrap(cls, shadow: ShadowDataCatalog):
        result = cls.__new__(cls)
        super(cls, result).__init__()
        result.shadow = shadow
        return result

    @property
    def updated(self) -> Signal:
        """``Signal[]`` Сигнал об изменении количества объектов."""
        return self.shadow.updated

    @property
    def added(self) -> Signal:
        """``Signal[str]`` Сигнал о добавлении объекта."""
        return self.shadow.added

    @property
    def removed(self) -> Signal:
        """``Signal[str]`` Сигнал об удалении объекта."""
        return self.shadow.removed

    @property
    def count(self) -> int:
        """Количество объектов данных."""
        return self.__len__()

    def __len__(self) -> int:
        return self.shadow.count()

    def add(self, data_object: DataObject):
        """Добавляет объект данных в хранилище.

        Args:
            data_object: Объект данных для добавления.
        """
        if data_object is not None:
            self.shadow.add(data_object.shadow)
        else:
            raise ValueError("None is not allow.")

    def remove(self, data_object: DataObject):
        """Удаляет объект данных.

        Объект данных при этом закрывается.

        Args:
            data_object: Объект данных для удаления.
        """
        if (isinstance(data_object, str)):
            data_object = self.find(data_object)
        if data_object is None:
            return
        self.shadow.remove(data_object.shadow)

    def remove_all(self):
        """Удаляет все объекты данных."""
        for obj in self.objects:
            self.remove(obj)

    def find(self, name: str) -> Optional[DataObject]:
        """Производит поиск объект данных по имени.

        Args:
            name: Имя объекта данных.

        Returns: 
            Искомый объект данных или None.
        """
        for obj in self.objects:
            if obj.name != name:
                continue
            return obj
        return None

    def exists(self, obj: DataObject) -> bool:
        """Проверяет присутствует ли объект в каталоге. Проверяет так-же и скрытые объекты, которые отсутствуют в общем списке.
        
        Args:
            obj: проверяемый объект данных.
        """
        return self.shadow.exists(obj.shadow)

    @property
    def objects(self) -> List[DataObject]:
        """Список объектов."""
        return [DataObject._wrap(obj) for obj in self.shadow.dataObjects()]

    @property
    def all_objects(self) -> List[DataObject]:
        """Список всех объектов, включая скрытые."""
        return [DataObject._wrap(obj) for obj in self.shadow.dataObjects(True)]

    def __is_table(self, obj):
        return isinstance(obj, Table)

    @property
    def tables(self) -> List[Table]:
        """Список таблиц."""
        return list(filter(self.__is_table, self.objects))

    @property
    def __all_tables(self) -> List[Table]:
        return list(filter(self.__is_table, self.all_objects))

    def __query_internal(self, query_text: str, is_hidden, dialect: str = None) -> Optional[Table]:
        from .providers import opener_instance
        result = opener_instance.query(query_text, *self.__all_tables, dialect=dialect)
        if is_hidden:
            result.shadow.setIsHidden(True)
        if result is not None:
            self.add(result)
        return result

    def query(self, query_text: str, dialect: str = None) -> Optional[Table]:
        """Выполняет SQL-запрос к перечисленным таблицам.

        Args:
            query_text: Текст запроса.
            dialect: Диалект, который используется при выполнении запроса. Допустимые значения `axioma` и `sqlite`.
              Значение по умолчанию установлено как значение свойства :attr:`sql_dialect`.

        Returns:
            Таблица, если результатом запроса является таблица.

        Raises:
            RuntimeError: При возникновении ошибки.

        Пример::

                query_text = "SELECT * FROM world, caps WHERE world.capital = caps.capital"
                joined = catalog.query(query_text)
        """
        return self.__query_internal(query_text, False, dialect)

    def query_hidden(self, query_text: str, dialect: str = None) -> Optional[Table]:
        """Выполняет SQL-запрос к таблицам. В отличие от :meth:`query` результирующий объект :class:`Table`
        добавляется в каталог как скрытый объект. Он не учитывается в общем списке и от него из этого каталога
        не приходят события. 

        Args:
            query_text: Текст запроса.
            dialect: Диалект, который используется при выполнении запроса. Допустимые значения `axioma` и `sqlite`.
              Значение по умолчанию установлено как :attr:`sql_dialect`.

        Returns:
            Таблица, если результатом запроса является таблица.
        """
        return self.__query_internal(query_text, True, dialect)

    @property
    def selection(self) -> Optional[Table]:
        """Таблица выборки, если она существует.
        
        See also:
            :attr:`axipy.gui.selection_manager`
        """
        return DataObject._wrap(self.shadow.selectionTable())

    def _init_instance(self, shadow: ShadowDataCatalog):
        self.shadow = shadow

    def __iter__(self):
        return (t for t in self.objects)

    @property
    def sql_dialect(self) -> str:
      """Тип используемого диалекта по умолчанию для выполнения SQL-предложений.

      Returns:
        Строковое значение, определяющее его тип. Возможные значения: `axioma` или `sqlite`.
      """
      from .providers import opener_instance
      return opener_instance.sql_dialect
