import abc
import typing
import pathlib
from configparser import ConfigParser


__all__ = [
    'Metadata',
    'LocationPath',
    'IParser',
    'MANIFEST_FILE_NAME',
    'ENTRY_POINT',
]


_ROOT_SECTION_NAME = 'general'
MANIFEST_FILE_NAME = 'manifest.ini'
ENTRY_POINT = '__init__.py'



Metadata = typing.NewType('Metadata', typing.Dict)
LocationPath = typing.NewType('LocationPath', typing.Union[str, pathlib.Path])


def read_manifest_config(manifest) -> dict:
    assert _ROOT_SECTION_NAME in manifest
    result = dict(manifest.items(_ROOT_SECTION_NAME))
    general_section = manifest[_ROOT_SECTION_NAME]
    assert 'name' in general_section
    assert 'description' in general_section
    names = {'default': general_section['name']}
    descriptions = {'default': general_section['description']}
    if 'i18n' in manifest:
        section = manifest['i18n']
        for key, value in section.items():
            if key.startswith('name_'):
                suffix = key[len('name_'):]
                names[suffix] = value
            if key.startswith('description_'):
                suffix = key[len('description_'):]
                descriptions[suffix] = value
    result['name'] = names
    result['description'] = descriptions
    if 'autoload' in general_section:
        result['autoload'] = general_section.getboolean('autoload')
    if 'system' in general_section:
        result['system'] = general_section.getboolean('system')
    return result


class IParser(abc.ABC):
    @classmethod
    def create(cls):
        return cls()

    @abc.abstractmethod
    def parse(self, location_path: LocationPath) -> Metadata:
        raise NotImplementedError

    @abc.abstractstaticmethod
    def supports(location_path: LocationPath) -> bool:
        return False

    def read_config(self, config_fileobject) -> dict:
        import itertools
        config = ConfigParser(strict=False)
        config.read_file(itertools.chain([f'[{_ROOT_SECTION_NAME}]'], config_fileobject))
        return read_manifest_config(config)
