
from typing import List

import axipy
from axipy.interface import AxiomaInterface
from axipy.axioma_plugin import AxiomaPlugin
from axipy.menubar import ToolButton, ActionButton
from .extension import UnloadActionInterface, UnloadQtAction
from axipy.da import DefaultKeys, state_manager
from axipy.menubar import Position

from .convexhull import ConvexHullAction
from .copy_paste_geometry import (
    GeometryStorage,
    register_paste_geometry_observer,
    paste_geometry_observer_key,
    PasteGeometry,
    CopyGeometry)
from .copy_paste_style import (
    StyleStorage,
    CopyStyle,
    PasteStyle,
    paste_style_observer_key,
    register_paste_style_observer)
from .create_outline import CreateOutline
from .mirror import HorizontalMirror, VerticalMirror
from .perpendicular import PerpendicularTool
from .right_angle import RightAngleTool
from .redirect import RedirectLine
from .voronoi import PolygonsVoronogo
from .cad_polyline.main import CadPolyline
from .arc_by_points import ArcByPoints
from .circle_by_points import CircleByPoints


class Plugin(AxiomaPlugin):
    def load(self):
        self.__unload_actions = []  # type: List[UnloadActionInterface]
        self.load_acitons()

    def load_acitons(self):
        self.position = Position('Дополнительно', 'Инструменты')

        self.init_perpendicular()
        self.init_convexhull()
        self.init_copy_paste_geometry()
        self.init_copy_paste_style()
        self.init_outline()
        self.init_mirror()
        self.init_redirect()
        self.init_voronoi()
        self.init_cad_polyline()
        self.init_right_angle()
        self.init_arc_by_points()
        self.init_circle_by_points()

    def unload(self):
        for action in self.__unload_actions:
            action.unload()

    def init_convexhull(self):
        title = self.tr('Оконтурить')
        convexhull = ConvexHullAction(title, self)
        button = self.create_action(
            title=title,
            on_click=lambda: convexhull.on_triggered(),
            enable_on=DefaultKeys.SelectionEditable,
            icon=self.local_file(
                'images',
                '24px',
                'convexhull.png'),
            tooltip = self.tr('Позволяет создавать контур (полигон) вокруг выбранных объектов.'),
            doc_file='convexhull.html')
        self.__unload_actions.append(UnloadQtAction(button.action, self))
        self.position.add(button, size=1)

    def init_copy_paste_geometry(self):
        storage = GeometryStorage()
        copy_title = self.tr("Копировать геометрию")
        cad_action = CopyGeometry(copy_title, self, storage)
        copy_button = self.create_action(
            title=copy_title,
            on_click=lambda: cad_action.on_triggered(),
            enable_on=state_manager.Selection,
            icon=self.local_file('images', '24px', 'copy_geometry.png'),
            tooltip = self.tr('Копирует геометрию в буфер обмена.'),
            doc_file='copy_paste_geometry.html')
        self.__unload_actions.append(
            UnloadQtAction(
                copy_button.action,
                self))
        self.position.add(copy_button, size=1)

        paste_title = self.tr("Вставить геометрию")
        def tool_factory(): return PasteGeometry(paste_title, self, storage)
        # Регистрируем наблюдатель
        register_paste_geometry_observer(storage)
        paste_button = self.create_tool(
            title=paste_title,
            on_click=tool_factory,
            enable_on=paste_geometry_observer_key,
            icon=self.local_file('images', '24px', 'paste_geometry.png'),
            tooltip = self.tr('Вставляет геометрию из буфера обмена в указанное место.'),
            doc_file='copy_paste_geometry.html')
        self.__unload_actions.append(
            UnloadQtAction(
                paste_button.action,
                self))
        self.position.add(paste_button, size=1)

    def init_copy_paste_style(self):
        storage = StyleStorage()
        copy_title = self.tr("Копировать стиль")
        copy_action = CopyStyle(copy_title, self, storage)
        copy_button = self.create_action(
            title=copy_title,
            on_click=lambda: copy_action.on_triggered(),
            enable_on=DefaultKeys.Selection,
            icon=self.local_file('images', '24px', 'copy_style.png'), 
            tooltip = self.tr('Копирует стиль оформления объекта в буфер обмена.'),
            doc_file='copy_paste_style.html')
        self.__unload_actions.append(
            UnloadQtAction(
                copy_button.action,
                self))
        self.position.add(copy_button, size=1)

        paste_title = self.tr("Вставить стиль")
        paste_action = PasteStyle(paste_title, self, storage)
        # Регистрируем наблюдателя
        register_paste_style_observer(storage)
        paste_button = self.create_action(
            title=paste_title,
            on_click=lambda: paste_action.on_triggered(),
            enable_on=paste_style_observer_key,
            icon=self.local_file('images', '24px', 'paste_style.png'),
            tooltip = self.tr('Применяет скопированный ранее стиль на выделенные объекты.'),
            doc_file='copy_paste_style.html')
        self.__unload_actions.append(
            UnloadQtAction(
                paste_button.action,
                self))
        self.position.add(paste_button, size=1)

    def init_outline(self):
        title = self.tr('Создание контура')
        observer_id = DefaultKeys.SelectionEditable
        create_ounline = CreateOutline(title, self, observer_id)
        button = self.create_action(
            title=title,
            on_click=lambda: create_ounline.on_triggered(),
            enable_on=observer_id,
            icon=self.local_file('images', '24px', 'create_outline.png'),
            tooltip = self.tr('Позволяет создавать вогнутый контур вокруг выбранных объектов.'),
            doc_file='create_outline.html')
        self.__unload_actions.append(UnloadQtAction(button.action, self))
        self.position.add(button, size=1)

    def init_mirror(self):
        vertical_title = self.tr("Отразить по вертикали")
        vertical_mirror = VerticalMirror(vertical_title, self)
        vertical_button = self.create_action(
            title=vertical_title,
            on_click=lambda: vertical_mirror.on_triggered(),
            enable_on=DefaultKeys.SelectionEditableIsSame,
            icon=self.local_file(
                'images',
                '24px',
                'vertical_mirror.png'),
            tooltip = self.tr('Объекты зеркально отражаются относительно центроида описанного вокруг них прямоугольника.'),
            doc_file='vertical_mirror.html')
        self.__unload_actions.append(
            UnloadQtAction(
                vertical_button.action,
                self))
        self.position.add(vertical_button, size=1)

        horizontal_title = self.tr("Отразить по горизонтали")
        horizontal_mirror = HorizontalMirror(horizontal_title, self)
        horizontal_button = self.create_action(
            title=horizontal_title,
            on_click=lambda: horizontal_mirror.on_triggered(),
            enable_on=DefaultKeys.SelectionEditableIsSame,
            icon=self.local_file(
                'images',
                '24px',
                'horizontal_mirror.png'),
            tooltip = self.tr('Объекты зеркально отражаются относительно центроида описанного вокруг них прямоугольника.'),
            doc_file='horizontal_mirror.html')
        self.__unload_actions.append(
            UnloadQtAction(
                horizontal_button.action,
                self))
        self.position.add(horizontal_button, size=1)

    def init_perpendicular(self):
        button = self.create_tool(
            title=self.tr('Перпендикуляр'),
            on_click=PerpendicularTool,
            icon=self.local_file('images', '24px', 'perpendicular.png'),
            tooltip = self.tr('Создаёт новый линейный объект перпендикулярный сегменту объекта.'),
            doc_file='perpendicular.html')
        self.__unload_actions.append(UnloadQtAction(button.action, self))
        self.position.add(button, size=1)

    def init_right_angle(self):
        button = self.create_tool(
            title=self.tr('Прямой угол'),
            on_click=RightAngleTool,
            icon=self.local_file('images', '24px', 'right_angle.png'),
            tooltip = self.tr('Создаёт новый линейный объект перпендикулярный сегменту объекта.'),
            doc_file='right_angle.html')
        self.__unload_actions.append(UnloadQtAction(button.action, self))
        self.position.add(button, size=1)

    def init_redirect(self):
        title = self.tr("Изменить направление линии")
        redirect_line = RedirectLine(title, self)
        button = self.create_action(
            title=title,
            on_click=lambda: redirect_line.on_triggered(),
            enable_on=DefaultKeys.SelectionEditableIsSame,
            icon=self.local_file(
                'images',
                '24px',
                'redirect.png'),
            tooltip = self.tr('Изменяет направление элементов выбранных объектов.'),
            doc_file='redirect.html')
        self.__unload_actions.append(UnloadQtAction(button.action, self))
        self.position.add(button, size=1)

    def init_voronoi(self):
        title = self.tr("Полигоны Вороного")
        observer_id = DefaultKeys.SelectionEditable
        polygons_voronogo = PolygonsVoronogo(title, self, observer_id)
        button = self.create_action(
            title=title,
            on_click=lambda: polygons_voronogo.on_triggered(),
            enable_on=observer_id,
            icon=self.local_file('images', '24px', 'voronoi.png'),
            tooltip = self.tr('Построение полигонов Вороного по выбранным объектам.'),
            doc_file='voronoi.html')
        self.__unload_actions.append(UnloadQtAction(button.action, self))
        self.position.add(button, size=1)

    def init_cad_polyline(self):
        title = axipy.tr("Создать по углу и расстоянию")
        observer_id = DefaultKeys.Editable
        def tool_factory(): return CadPolyline(title, self, observer_id)
        button = self.create_tool(title, on_click=tool_factory, 
            icon=self.local_file('images', '24px', 'cad_polyline.png'),
            enable_on=observer_id)
        self.__unload_actions.append(UnloadQtAction(button.action, self))
        self.position.add(button, size=1)
    
    def init_arc_by_points(self):
        title = axipy.tr("Дуга по 2 или 3 точкам")
        observer_id = DefaultKeys.Editable
        def tool_factory(): return ArcByPoints(self, title, observer_id)
        button = self.create_tool(title, on_click=tool_factory, \
            icon=self.local_file('images', '24px', 'arc_by_points.png'),
            enable_on=observer_id)
        self.__unload_actions.append(UnloadQtAction(button.action, self))
        self.position.add(button, size=1)

    def init_circle_by_points(self):
        title = axipy.tr("Окружность по трём точкам")
        observer_id = DefaultKeys.Editable
        def tool_factory(): return CircleByPoints(self, title)
        button = self.create_tool(title=title, on_click=tool_factory, \
            icon=self.local_file('images', '24px', 'circle_by_points.png'), \
            enable_on=observer_id)
        self.__unload_actions.append(UnloadQtAction(button.action, self))
        self.position.add(button, size=1)

