"""Package for parsing axipy plugins' metadata.

Does not depend on axipy itself.
"""

import os
import os.path
from typing import Union
import zipfile
import configparser


__all__ = [
    'read_metadata_params',
    'read_metadata',
    'install',
    'MANIFEST_FILE_NAME',
    'ARCHIVE_EXTENSION',
]


MANIFEST_FILE_NAME = 'manifest.ini'
ARCHIVE_EXTENSION = 'axp'
_ARCHIVE_SUFFIX = '.' + ARCHIVE_EXTENSION
_ENTRY_POINT = '__init__.py'


def read_manifest_config(manifest) -> dict:
    result = dict(manifest.items('general'))
    assert 'general' in manifest
    general_section = manifest['general']
    assert 'name' in general_section
    assert 'description' in general_section
    names = {'default': general_section['name']}
    descriptions = {'default': general_section['description']}
    if 'i18n' in manifest:
        section = manifest['i18n']
        for key, value in section.items():
            if key.startswith('name_'):
                suffix = key[len('name_'):]
                names[suffix] = value
            if key.startswith('description_'):
                suffix = key[len('description_'):]
                descriptions[suffix] = value
    result['name'] = names
    result['description'] = descriptions
    if 'autoload' in general_section:
        result['autoload'] = general_section.getboolean('autoload')
    if 'system' in general_section:
        result['system'] = general_section.getboolean('system')
    return result


def read_manifest_file(manifest_file: Union[str, bytes, os.PathLike]) -> dict:
    manifest = configparser.ConfigParser()
    manifest.read(manifest_file, encoding='utf-8')
    return(read_manifest_config(manifest))


def read_metadata_folder(plugin_dir: Union[str, bytes, os.PathLike]) -> dict:
    manifest_file = os.path.join(plugin_dir, MANIFEST_FILE_NAME)
    result = read_manifest_file(manifest_file)
    doc_dir = os.path.join(plugin_dir, 'documentation')
    if os.path.isdir(doc_dir):
        def is_doc_entry(filename: str): return filename.startswith(
            'index') and filename.endswith('html')
        doc_entry_points = filter(is_doc_entry, os.listdir(doc_dir))
        result['documentation'] = [os.path.join(
            doc_dir, entry) for entry in doc_entry_points]
    return result


def install(package_file, destination):
    with zipfile.ZipFile(package_file, 'r') as package:
        package.extractall(destination)


def read_metadata_package(package_file: Union[str, bytes, os.PathLike]) -> dict:
    import codecs
    from pathlib import PurePosixPath
    with zipfile.ZipFile(package_file) as archive:
        def is_root_content(path):
            return (path.endswith('/') and path.count('/') == 1) or '/' not in path
        root_content = set(filter(is_root_content, archive.namelist()))
        if (len(root_content) != 1
                or not next(iter(root_content)).endswith('/')):
            raise RuntimeError('Archive must contain one root directory')
        plugin_dir = PurePosixPath(next(iter(root_content)))
        try:
            entry_point = plugin_dir / _ENTRY_POINT
            entry_point_info = archive.getinfo(str(entry_point))
        except KeyError:
            raise RuntimeError(f'Module must contain an entry point: {_ENTRY_POINT}')
        try:
            manifest_path = plugin_dir / MANIFEST_FILE_NAME
            manifest_path_info = archive.getinfo(str(manifest_path))
        except KeyError:
            raise RuntimeError('Module must contain a manifest file')
        plugin_id = plugin_dir.name
        with archive.open(manifest_path_info) as manifest_file:
            config = configparser.ConfigParser()
            config.read_file(codecs.getreader('utf-8')(manifest_file))
            result = read_manifest_config(config)
            result['id'] = plugin_id
            return result


def read_metadata_params(plugin_path: Union[str, bytes, os.PathLike]) -> dict:
    if plugin_path.endswith(_ARCHIVE_SUFFIX):
        return read_metadata_package(plugin_path)
    else:
        return read_metadata_folder(plugin_path)


def read_metadata(plugin_path: Union[str, bytes, os.PathLike]) -> str:
    import json
    params = read_metadata_params(plugin_path)
    return json.dumps(params)
