from .interface import AxiomaInterface
from axipy.menubar import Button, ActionButton, ToolButton, Position
from typing import Union, Any, Callable
from PySide2.QtGui import QIcon
from PySide2.QtWidgets import QAction
from axipy.da import DefaultKeys, state_manager
from axipy.gui import gui_instance, MapTool


__all__ = [
    'AxiomaPlugin',
]


class AxiomaPlugin(AxiomaInterface):
    """Модуль для ГИС "Аксиома".

    Содержит вспомогательные функции и свойства, которые могут быть использованы 
    при реализации пользовательского модуля.

    Note:
        Не переопределяйте конструктор. Переопределяйте метод :meth:`load`.

    See also:
        Подробнее в главе :ref:`to_modules`.
    """

    @classmethod
    def _axioma_plugin_create(cls, plugin_dir):
        result = cls()
        result.initialize(plugin_dir)
        result._axioma_plugin_load()
        return result

    def _axioma_plugin_load(self):
        self.load()

    def _axioma_plugin_unload(self):
        self.unload()

    def load(self):
        """Загружает модуль.

        Переопределяйте этот метод для задания логики модуля."""
        pass

    def unload(self):
        """Выгружает модуль.

        Переопределяйте этот метод для очистки ресурсов."""
        pass

    def create_tool(self, title: str, on_click: Callable[[], MapTool], icon: Union[str, QIcon] = '', enable_on: Union[str, DefaultKeys] = None) -> ToolButton:
        """Создает кнопку с инструментом.
        
        Args:
            title: Текст.
            on_click: Класс инструмента.
            icon: Иконка. Может быть путем к файлу или адресом ресурса.
            enable_on: Идентификатор наблюдателя для определения доступности кнопки.
        Return:
            Кнопка с инструментом.

        See also:
            class:`axipy.da.StateManager`.

        Note:
            То же, что и :class:`axipy.menubar.ToolButton`, но дополнительно
            делает идентификатор кнопки уникальным для данного модуля.
        """
        result = ToolButton(title, on_click, icon, enable_on)
        self._disambiguate(result)
        return result

    def __full_doc_file(self, filename) -> str:
        return 'file:///{}'.format(self.local_file('documentation', filename).replace('\\', '/'))

    def create_action(self, title: str, on_click: Callable[[], Any], icon: Union[str, QIcon] = '',
                      enable_on: Union[str, DefaultKeys] = None, tooltip: str = None, doc_file: str = None) -> ActionButton:
        """Создает кнопку с действием.
        
        Args:
            title: Текст.
            on_click: Действие на нажатие.
            icon: Иконка. Может быть путем к файлу или адресом ресурса.
            enable_on: Идентификатор наблюдателя для определения доступности кнопки.
            tooltip: Строка с дополнительной короткой информацией по данному действию.
            doc_file: Относительная ссылка на файл документации. Расположение рассматривается по отношению к
            каталогу documentation
        Return:
            Кнопка с действием.

        See also:
            :class:`axipy.da.StateManager`.

        Note:
            То же, что и :class:`axipy.menubar.ActionButton`, но дополнительно
            делает идентификатор кнопки уникальным для данного модуля.
        """
        result = ActionButton(title, on_click, icon, enable_on, tooltip = tooltip)
        if doc_file is not None:
            result.action.setWhatsThis(self.__full_doc_file(doc_file))
        self._disambiguate(result)
        return result

    def _disambiguate(self, button: Button):
        button.action.setObjectName(self._make_action_id(button.action.text()))

    def get_position(self, tab: str, group: str) -> Position:
        """Возвращает положение в меню. Может заранее не существовать.

        Args:
            tab: Название вкладки.
            group: Название группы.

        Return:
            Положение для кнопки.

        Note:
            Дублирует :class:`axipy.menubar.Position`.
        """
        return self.menubar.get_position(tab, group)
