import traceback

from PySide2.QtSql import QSqlQuery

from axipy import *
from axipy.app import Notifications
from axipy.concurrent import task_manager, AxipyProgressHandler, ProgressSpecification, ProgressGuiFlags

from .CoordinateProcess import *
from .LoadPointsDialog import *
from .SavePointsDialog import *

from PySide2.QtGui import QIcon
from PySide2.QtWidgets import QDialog

def ensure_sql_init_for_bg_thread():
    query = QSqlQuery(get_database())
    task_manager.run_in_gui(lambda: query.exec_("Select 'Init SQL Engine'"))

class Plugin(AxiomaPlugin):

    def load(self):
        position = Position('Дополнительно', 'Инструменты')
        self.loadTool = self.create_action(
            self.tr('Загрузить данные из точек'),
            icon=self.__get_icon('open.png'),
            on_click=self.load_points,
            enable_on=state_manager.Editable,
            tooltip = self.tr('Загружает геометрию объектов из таблицы в виде последовательности точек'),
            doc_file='index.html#load')
        position.add(self.loadTool)
        self.saveTool = self.create_action(
            self.tr('Сохранить в виде точек'),
            icon=self.__get_icon('save.png'),
            on_click=self.save_points,
            enable_on=state_manager.HasTables,
            tooltip = self.tr('Сохраняет геометрию объектов в виде таблицы как последовательность точек'),
            doc_file='index.html#unload')
        position.add(self.saveTool)

    def __get_icon(self, filename):
        return QIcon(os.path.join(os.path.dirname(__file__), filename))

    def unload(self):
        self.saveTool.remove()
        self.loadTool.remove()

    def __save_points( self, handler: AxipyProgressHandler, table_in: Table, \
        cw : CoordinateWriter, result_file_name: str):
        ensure_sql_init_for_bg_thread()
        definition = {
            'src': result_file_name,
            'schema': attr.schema(
                attr.integer(CoordinateProcess.IdFeatureField),
                attr.integer(CoordinateProcess.IdGeometryField),
                attr.integer(CoordinateProcess.IdPointField),
                attr.float(CoordinateProcess.ValueFirstField),
                attr.float(CoordinateProcess.ValueSecondField)
            ),
            'hidden': True
        }
        table_out = provider_manager.create(definition) # type: Table
        handler.set_max_progress(table_in.count())
        try:
            table_out.insert((Feature({CoordinateProcess.IdFeatureField: v[0],
                                CoordinateProcess.IdGeometryField: v[1],
                                CoordinateProcess.IdPointField: v[2],
                                CoordinateProcess.ValueFirstField: v[3],
                                CoordinateProcess.ValueSecondField: v[4]}) 
                    for v in cw.write(table_in.items(), handler)
                ))
            handler.prepare_to_write_changes()
            table_out.commit()
            table_out.close()
            self.notifications.push(self.tr('Информация'), self.tr('Данные сохранены.'))
        except RuntimeError:
            traceback.print_exc()
            if handler.is_canceled():
                self.notifications.push(self.tr('Предупреждение'), self.tr('Процесс прерван пользователем.'), Notifications.Warning)
            else:
                self.notifications.push(self.tr('Ошибка'), self.tr('Процесс завершен неудачно.'), Notifications.Critical)
            table_out.restore()
            table_out.close()

    def save_points(self):
        # Сохранить геометрию в виде табличных данных
        fd = SavePointsDialog(self)
        if fd.exec() == QDialog.Accepted:
            cw = CoordinateWriter()
            cw.SimpleCoordinates = fd.is_simple_coordinates()
            spec = ProgressSpecification(
                description='Экспорт данных',
                flags=ProgressGuiFlags.CANCELABLE)
            task_manager.run_and_get(
                spec,
                self.__save_points,
                fd.result_table(), 
                cw, 
                fd.result_file_name())

    def __load_points(self, handler: AxipyProgressHandler, reader: CoordinateReader,\
         qry_text, edit_layer, coordsystem):
        try:
          qry = data_manager.query(qry_text)
          # TODO: Можно сделать прогрессбар не бесконечным, если вначале сделать запрос
          # на общее число записей. Но есть проблема: Сейчас сигнал из
          # AxipyProgressHandler.add_progress() отправляется к прогрессу при каждом вызове
          # и для 10 - 100k точек это приводит к зависанию интерфейса и возрастанию
          # времени работы. После оптимизации отправки сигналов с прогрессом можно
          # этот код поправить
          features = [ Feature(geometry = g, style = Style.for_geometry(g))
                        for g in reader.read(qry.items(), handler, coordsystem) ]
          edit_layer.data_object.insert(features)
        finally:
            data_manager.remove(qry)
        if handler.is_canceled():
            self.notifications.push(self.tr('Предупреждение'), self.tr('Процесс прерван пользователем.'), Notifications.Warning)
        else:
            self.notifications.push(self.tr('Информация'), self.tr('Данные загружены.'))

    def load_points(self):
        # Зачитать геометрию из файла
        if isinstance(view_manager.active, MapView):
            edit_layer = view_manager.active.map.editable_layer
            if edit_layer is not None:
                fd = LoadPointsDialog(self, view_manager.active.coordsystem)
                if fd.exec() == QDialog.Accepted:
                    cr = CoordinateReader()
                    cr.SimpleCoordinates = fd.is_simple_coordinates()
                    spec = ProgressSpecification(
                        description='Импорт данных',
                        flags=ProgressGuiFlags.CANCELABLE)
                    query_text = fd.result_sql_text()
                    task_manager.run_and_get(
                        spec,
                        self.__load_points,
                        cr,
                        query_text,
                        edit_layer,
                        fd.result_coordsystem())

