from axipy.cpp_core_geometry import ShadowRectangle, ShadowRoundRectangle, ShadowEllipse, ShadowArc, ShadowText
from axipy.cs import CoordSystem
from axipy.da import Geometry
from axipy.utl import Rect, Pnt

from typing import Union, Tuple


class Rectangle(Geometry):
    """Геометрический объект типа прямоугольник.

    Args:
        par: Прямоугольник класса :class:`Rect` или перечень координат через запятую (`xmin`, `ymin`, `xmax`, `ymax`) или списком :class:`list`.
        cs: Система Координат, в которой создается геометрия.

    .. literalinclude:: /../../tests/doc_examples/test_example_geometry.py
        :caption: Пример.
        :pyobject: test_run_example_geometry_rectangle
        :lines: 2-
        :dedent: 4
    """

    def __init__(self, *par: Union[Rect, float], cs: CoordSystem = None):
        if len(par) == 1 and type(par[0]) == Rect:
            v = par[0]
        elif len(par) == 1 and type(par[0]) == list and len(par[0]) == 4:
            v = Rect(par[0][0], par[0][1], par[0][2], par[0][3])
        elif len(par) == 4:
            v = Rect(par[0], par[1], par[2], par[3])
        else:
            raise TypeError('Unsupported data type')
        self._set_shadow(ShadowRectangle(
            v.to_qt(), None if cs is None else cs.shadow))

    @property
    def xmin(self) -> float:
        """Минимальное значение X."""
        return self.shadow.get_rect().left()

    @xmin.setter
    def xmin(self, v: float):
        r = self.shadow.get_rect()
        r.setLeft(v)
        self.shadow.set_rect(r)

    @property
    def ymin(self) -> float:
        """Минимальное значение Y."""
        return self.shadow.get_rect().top()

    @ymin.setter
    def ymin(self, v: float):
        r = self.shadow.get_rect()
        r.setTop(v)
        self.shadow.set_rect(r)

    @property
    def xmax(self) -> float:
        """Максимальное значение X."""
        return self.shadow.get_rect().right()

    @xmax.setter
    def xmax(self, v: float):
        r = self.shadow.get_rect()
        r.setRight(v)
        self.shadow.set_rect(r)

    @property
    def ymax(self) -> float:
        """Максимальное значение Y."""
        return self.shadow.get_rect().bottom()

    @ymax.setter
    def ymax(self, v: float):
        r = self.shadow.get_rect()
        r.setBottom(v)
        self.shadow.set_rect(r)

    def __repr__(self):
        return '{} xmin={} ymin={} xmax={} ymax={}'.format(self._class_name, self.xmin, self.ymin, self.xmax, self.ymax) + self._cs_str


class RoundRectangle(Rectangle):
    """Геометрический объект типа скругленный прямоугольник.

    Args:
        rect: Прямоугольник класса :class:`Rect` или как :class:`list`.
        xRad: Скругление по X.
        yRad: Скругление по Y.
        cs: Система Координат, в которой создается геометрия.

    .. literalinclude:: /../../tests/doc_examples/test_example_geometry.py
        :caption: Пример.
        :pyobject: test_run_example_geometry_rrectangle
        :lines: 2-
        :dedent: 4
    """

    def __init__(self, rect: Union[Rect, list], xRad: float, yRad: float, cs: CoordSystem = None):
        if type(rect) == Rect:
            v = rect
        elif type(rect) == list and len(rect) == 4:
            v = Rect(rect[0], rect[1], rect[2], rect[3])
        else:
            raise TypeError('Unsupported data type')
        self._set_shadow(ShadowRoundRectangle(
            v.to_qt(), xRad, yRad, None if cs is None else cs.shadow))

    @property
    def xRadius(self) -> float:
        """Скругление углов по координате X."""
        return self.shadow.xRadius()

    @xRadius.setter
    def xRadius(self, v):
        self.shadow.setXRadius(v)

    @property
    def yRadius(self) -> float:
        """Скругление углов по координате Y."""
        return self.shadow.yRadius()

    @yRadius.setter
    def yRadius(self, v):
        self.shadow.setYRadius(v)

    def __repr__(self):
        return '{} xmin={} ymin={} xmax={} ymax={} xraduis={} yraduis={}'.format(self._class_name, self.xmin, self.ymin, self.xmax, self.ymax, self.xRadius, self.yRadius) + self._cs_str


class Ellipse(Geometry):
    """Геометрический объект типа эллипс.

    Args:
        rect: Прямоугольник класса :class:`Rect` или как :class:`list`.
        cs: Система Координат, в которой создается геометрия.

    .. literalinclude:: /../../tests/doc_examples/test_example_geometry.py
        :caption: Пример.
        :pyobject: test_run_example_geometry_ellipse
        :lines: 2-
        :dedent: 4
    """

    def __init__(self, rect: Union[Rect, list], cs: CoordSystem = None):
        if type(rect) == Rect:
            v = rect
        elif type(rect) == list and len(rect) == 4:
            v = Rect(rect[0], rect[1], rect[2], rect[3])
        else:
            raise TypeError('Unsupported data type')
        self._set_shadow(ShadowEllipse(
            v.to_qt(), None if cs is None else cs.shadow))

    @property
    def center(self) -> Pnt:
        """Центр эллипса."""
        return Pnt.from_qt(self.shadow.center())

    @center.setter
    def center(self, v: Union[Pnt, tuple]):
        self.shadow.setCenter(Pnt._fixPnt(v).to_qt())

    @property
    def majorSemiAxis(self) -> float:
        """Радиус большой полуоси эллипса."""
        return self.shadow.majorSemiAxis()

    @majorSemiAxis.setter
    def majorSemiAxis(self, v: float):
        self.shadow.setMajorSemiAxis(v)

    @property
    def minorSemiAxis(self) -> float:
        """Радиус малой полуоси эллипса."""
        return self.shadow.minorSemiAxis()

    @minorSemiAxis.setter
    def minorSemiAxis(self, v: float):
        self.shadow.setMinorSemiAxis(v)

    def __repr__(self):
        return '{} center={} major={} minor={}'.format(self._class_name, self.center, self.majorSemiAxis, self.minorSemiAxis) + self._cs_str


class Arc(Geometry):
    """Геометрический объект типа дуга.

    Args:
        rect: Прямоугольник класса :class:`Rect` или как :class:`list`.
        startAngle: Начальный угол дуги.
        endAngle: Конечный угол дуги.
        cs: Система Координат, в которой создается геометрия.

    .. literalinclude:: /../../tests/doc_examples/test_example_geometry.py
        :caption: Пример.
        :pyobject: test_run_example_geometry_arc
        :lines: 2-
        :dedent: 4
    """

    def __init__(self, rect: Union[Rect, list], startAngle: float, endAngle: float, cs: CoordSystem = None):
        if type(rect) == Rect:
            v = rect
        elif type(rect) == list and len(rect) == 4:
            v = Rect(rect[0], rect[1], rect[2], rect[3])
        else:
            raise TypeError('Unsupported data type')
        self._set_shadow(ShadowArc(v.to_qt(), startAngle,
                                   endAngle, None if cs is None else cs.shadow))

    @property
    def center(self) -> Pnt:
        """Центр дуги."""
        return Pnt.from_qt(self.shadow.center())

    @center.setter
    def center(self, v: Union[Pnt, tuple]):
        self.shadow.setCenter(Pnt._fixPnt(v).to_qt())

    @property
    def xRadius(self) -> float:
        """Радиус большой полуоси прямоугольника, в который вписана дуга."""
        return self.shadow.xRadius()

    @xRadius.setter
    def xRadius(self, v: float):
        self.shadow.setXRadius(v)

    @property
    def yRadius(self) -> float:
        """Радиус малой полуоси прямоугольника, в который вписана дуга."""
        return self.shadow.yRadius()

    @yRadius.setter
    def yRadius(self, v: float):
        self.shadow.setYRadius(v)

    @property
    def startAngle(self) -> float:
        """Начальный угол дуги."""
        return self.shadow.startAngle()

    @startAngle.setter
    def startAngle(self, v: float):
        self.shadow.setStartAngle(v)

    @property
    def endAngle(self) -> float:
        """Конечный угол дуги."""
        return self.shadow.endAngle()

    @endAngle.setter
    def endAngle(self, v: float):
        self.shadow.setEndAngle(v)

    def __repr__(self):
        return '{} center={} xradius={} yradius={} start={} end={}'.format(self._class_name, self.center, self.xRadius, self.yRadius, self.startAngle, self.endAngle)  + self._cs_str


class Text(Geometry):
    """Геометрический объект типа текст. 

    Warning:
        Геометрия текста, и, в отличие от остальных типов объектов, определяется так же и стилем его оформления :class:`axipy.da.TextStyle`.

    Args:
        text: Текст
        topLeft: Точка привязки. Допустимо задание точки :class:`Pnt` или как пара :class:`tuple`.
        cs: Система Координат, в которой создается геометрия.

    .. literalinclude:: /../../tests/doc_examples/test_example_geometry.py
        :caption: Пример.
        :pyobject: test_run_example_geometry_text
        :lines: 2-
        :dedent: 4
    """

    def __init__(self, text: str, point: Union[Pnt, tuple], cs: CoordSystem = None):
        if isinstance(point, Pnt):
            v = point
        elif type(point) == tuple and len(point) == 2:
            v = Pnt(point[0], point[1])
        else:
            raise TypeError('Unsupported data type')
        self._set_shadow(ShadowText(
            text, v.to_qt(), None if cs is None else cs.shadow))

    @property
    def startPoint(self) -> Pnt:
        """Координаты точки привязки."""
        return Pnt.from_qt(self.shadow.startPoint())

    @startPoint.setter
    def startPoint(self, v: Union[Pnt, tuple]):
        self.shadow.setStartPoint(Pnt._fixPnt(v).to_qt())

    @property
    def text(self) -> str:
        """Текст."""
        return self.shadow.text()

    @text.setter
    def text(self, v: str):
        self.shadow.setText(v)

    @property
    def angle(self) -> float:
        """Угол поворота текста."""
        return self.shadow.angle()

    @angle.setter
    def angle(self, v: float):
        self.shadow.setAngle(v)

    def __repr__(self):
        return '{} text={} point={} angle={}'.format(self._class_name, self.text, self.startPoint, self.angle) + self._cs_str
