from typing import List, Optional, Any, Union
from axipy.cpp_core_core import ShadowIO
from axipy.cpp_core_dp import query as cpp_query
from axipy.da import DataObject, Table, Schema
from axipy.decorator import InitOnce


class IOClass:
    """Класс открытия/создания объектов данных.

    Attention:
        Устарел в версии 3.0.0. Используйте :class:`axipy.da.ProviderManager` и
        его готовый экземпляр :attr:`axipy.da.provider_manager`.

    Note:
        Используйте готовый экземпляр этого класса :attr:`axipy.io`.

    Пример::

        from axipy import io
        table = provider_manager.openfile('../path/to/datadir/table.tab')
    """

    def __init__(self):
        self.s = None

    def _service(self):
        if self.s is None:
            try:
                from axipy.app import core_instance
            except ImportError:
                raise RuntimeError('axipy is not initialized')
            self.s = ShadowIO(core_instance)
        return self.s

    @property
    def shadow(self):
        return self._service()

    @staticmethod
    def _wrap(obj) -> DataObject:
        return DataObject._wrap(obj)

    def loaded_providers(self) -> dict:
        """Возвращает список всех загруженных провайдеров данных.

        Returns:
            Провайдеры в виде пар ``(Идентификатор : Описание)``.

        ::

            print(provider_manager.loaded_providers())
        """
        return self._service().providersInfo()

    def openfile(self, filepath: str, dataobject: str = '', provider: str = '') -> DataObject:
        """Открывает данные из файла.

        Args:
            filepath: Путь к открываемому файлу.
            dataobject: Имя открываемого объекта данных.
            provider: Провайдер, с помощью которого будет произведено открытие. Если не указан, выбирается наиболее подходящий.
                Перечень можно получить с помощью :meth:`loaded_providers`

        Пример::

            table = provider_manager.openfile('../path/to/datadir/example.gpkg', dataobject='tablename', provider='SqliteDataProvider')
        """
        result = self._service().openfile(filepath, dataobject, provider)
        return self._wrap(result)

    def open(self, definition: dict) -> DataObject:
        """Открывает данные по описанию.

        Формат описания объектов данных индивидуален для каждого провайдера данных,
        однако многие элементы используются для всех провайдеров данных.

        Args:
            definition: Описание объекта данных.

        Пример::

            # Пример открытия GPKG файла::
            definition = { 'src': '../path/to/datadir/example.gpkg', 
                           'dataobject': 'tablename',
                           'provider': 'SqliteDataProvider' }
            table = provider_manager.open(definition)

        Пример открытия таблицы базы данных::

            definition = {"src": "localhost",
                          "db": "sample",
                          "user": "postgres",
                          "password": "postgres",
                          "dataobject": "public.world",
                          "provider": "PgDataProvider"}
            table = provider_manager.open(definition)

        """
        result = self._service().open(definition)
        return self._wrap(result)

    def create(self, definition: dict) -> DataObject:
        """Создает данные из описания.

        Возможные параметры:
            - ``src`` - Строка, определяющая местоположение источника данных. Это может быть либо путь к файлу с расширением TAB, либо пустая строка (для таблицы, размещаемой в памяти).
            - ``schema`` - Схема таблицы. Задается массивом объектов, содержащих атрибуты.

        Args:
            definition: Описание объекта данных.

        Пример::

            definition = {
                'src': '../path/to/datadir/edit/table.tab',
                'schema': attr.schema(
                    attr.string('field1'),
                    attr.integer('field2'),
                ),
            }
            table = provider_manager.create(definition)
        """
        if 'schema' in definition:
            if isinstance(definition['schema'], Schema):
                definition['schema'] = definition['schema'].to_dict()
        result = self._service().create(definition)
        return self._wrap(result)

    def createfile(self, filepath: str, schema: Schema):
        """Создает таблицу.

        :meth:`create` выполняет ту же функцию, но в более обобщенном виде.

        Args:
            filepath: Путь к создаваемой таблице.
            schema: Схема таблицы.
        """
        return self.create({'src': filepath, 'schema': schema})

    def read_contents(self, definition: Union[dict, str]) -> List[str]:
        """Читает содержимое источника данных.

        Обычно используется для источников, способных содержать несколько
        объектов данных.

        Args:
            definition: Описание источника данных.

        Returns:
            Имена объектов данных.

        Пример::

            >>> provider_manager.read_contents('../path/to/datadir/example.gpkg')
            ['world', 'worldcap']

            >>> world = provider_manager.openfile('../path/to/datadir/example.gpkg',
            ...         dataobject='world')
        """
        if isinstance(definition, str):
            definition = {'src': definition}
        return self._service().read_contents(definition)

    def query(self, query_text: str, *tables) -> Optional[Table]:
        """Выполняет SQL-запрос к перечисленным таблицам.

        Args:
            query_text: Текст запроса.
            *tables: Список таблиц, к которым выполняется запрос. 

        Returns:
            Таблица, если результатом запроса является таблица.

        Raises:
            RuntimeError: При возникновении ошибки.

        Пример::

                query_text = "SELECT * FROM world, caps WHERE world.capital = caps.capital"
                joined = provider_manager.query(query_text, world, caps)
        """
        return self._wrap(cpp_query(query_text, [t.shadow for t in tables]))

    @staticmethod
    def _query_result(query_table: Table) -> Any:
        feature = next(query_table.items())
        return tuple(v for v in feature)

    def _query_aggregate(self, query_text: str, *tables) -> tuple:
        """Выполняет SQL-запрос и возвращает значения первой записи.

        Удобен при чтении результата агрегирующих запросов.

        Args:
            query_text: Текст запроса.
            *tables: Список таблиц, к которым выполняется запрос.

        Returns:
            Кортеж значений.

        Пример::

            avg, max, sum = provider_manager.query_aggregate(f'SELECT AVG(myvalue), MAX(myvalue), SUM(myvalue) FROM {table.name}', table)
        """
        query_table = self.query(query_text, *tables)
        return self._query_result(query_table)

    @property
    def sql_dialect(self) -> str:
      """Возвращает тип используемого решения для выполнения SQL-предложений

      Returns:
        Строковое значение, определяющее тип. Возможные значения - `axioma` или `sqlite`.
      """
      return self._service().currentSqlDialect()


opener_instance = IOClass()
