import os.path
from typing import Optional, List, Union
from .data_provider import DataProvider
from .tab_data_provider import TabDataProvider
from .shp_data_provider import ShapeDataProvider
from .csv_data_provider import CsvDataProvider
from .mif_data_provider import MifMidDataProvider
from .excel_data_provider import ExcelDataProvider
from .sqlite_data_provider import SqliteDataProvider
from .postgre_data_provider import PostgreDataProvider
from .mssql_data_provider import MsSqlDataProvider
from .oracle_data_provider import OracleDataProvider
from .rest_data_provider import RestDataProvider
from .gdal_data_provider import GdalDataProvider
from .tms_data_provider import TmsDataProvider
from .wmts_data_provider import WmtsDataProvider
from .generic_provider import GenericDataProvider, GenericSource, GenericDestination
from .source import Source, Destination
from axipy.cpp_core_dp import Converter
from .opener import opener_instance
from axipy.da import DataObject, Table


class ProviderManager:
    """Класс открытия/создания объектов данных.

    Note:
        Используйте готовый экземпляр этого класса :attr:`axipy.da.provider_manager`.

    Note:
        Для удобного задания параметров используйте экземпляры провайдеров:
        :attr:`tab`, :attr:`shp`, :attr:`csv`, :attr:`mif`, :attr:`excel`,
        :attr:`sqlite`, :attr:`postgre`, :attr:`oracle`, :attr:`mssql`.

    Note:
        Открытые данные автоматически попадают в хранилище данных
        :class:`axipy.da.DataManager`.

    Пример открытия локальной таблицы::

        table = provider_manager.openfile('../path/to/datadir/table.tab')
    """

    def _init_instance(self):
        from axipy.app import core_instance
        known_types = [
            TabDataProvider, ShapeDataProvider, CsvDataProvider,
            MifMidDataProvider, ExcelDataProvider, SqliteDataProvider,
            PostgreDataProvider, OracleDataProvider, MsSqlDataProvider,
            RestDataProvider, GdalDataProvider, TmsDataProvider, WmtsDataProvider
        ]
        providers = []
        infos = Converter.providersInfo()
        for info in infos:
            id = info['id']
            prov_type = next(filter(lambda p: p._identifier ==
                                    id, known_types), GenericDataProvider)
            providers.append(prov_type(info))
        self.providers = providers

    @property
    def tab(self) -> TabDataProvider:
        """Провайдер MapInfo."""
        return self._find_by_type(TabDataProvider)

    @property
    def shp(self) -> ShapeDataProvider:
        """Векторный провайдер OGR."""
        return self._find_by_type(ShapeDataProvider)

    @property
    def csv(self) -> CsvDataProvider:
        """Файловый провайдер - Текст с разделителями."""
        return self._find_by_type(CsvDataProvider)

    @property
    def mif(self) -> MifMidDataProvider:
        """Провайдер данных MIF-MID."""
        return self._find_by_type(MifMidDataProvider)

    @property
    def excel(self) -> ExcelDataProvider:
        """Провайдер чтения файлов Excel."""
        return self._find_by_type(ExcelDataProvider)

    @property
    def sqlite(self) -> SqliteDataProvider:
        """Векторный провайдер sqlite."""
        return self._find_by_type(SqliteDataProvider)

    @property
    def postgre(self) -> PostgreDataProvider:
        """Провайдер для базы данных PostgreSQL."""
        return self._find_by_type(PostgreDataProvider)

    @property
    def mssql(self) -> MsSqlDataProvider:
        """Провайдер для базы данных MSSQLServer."""
        return self._find_by_type(MsSqlDataProvider)

    @property
    def oracle(self) -> OracleDataProvider:
        """Провайдер для базы данных Oracle."""
        return self._find_by_type(OracleDataProvider)

    @property
    def rest(self) -> RestDataProvider:
        """Провайдер REST."""
        return self._find_by_type(RestDataProvider)

    @property
    def gdal(self) -> GdalDataProvider:
        """Растровый провайдер GDAL."""
        return self._find_by_type(GdalDataProvider)

    @property
    def tms(self) -> TmsDataProvider:
        """Тайловый провайдер."""
        return self._find_by_type(TmsDataProvider)

    @property
    def wmts(self) -> WmtsDataProvider:
        """Web Map Tile Service."""
        return self._find_by_type(WmtsDataProvider)

    def _find_by_id(self, id: str) -> DataProvider:
        for provider in self.providers:
            if provider.id == id:
                return provider
        return None

    def find_by_extension(self, file_extension: str) -> DataProvider:
        for provider in self.providers:
            if file_extension.lower() in [ext.lower() for ext in provider.file_extensions()]:
                return provider
        raise ValueError(
            f'Could not find provider for file extension: {file_extension}')

    def _find_by_type(self, typedef) -> DataProvider:
        for provider in self.providers:
            if type(provider) is typedef:
                return provider
        raise RuntimeError(f'Could not find provider of type {typedef}.')

    def openfile(self, filepath: str, *args, **kwargs) -> DataObject:
        """Открывает данные из файла.

        Args:
            filepath: Путь к открываемому файлу.
            **kwargs: Именованные аргументы.

        Пример::

            table = provider_manager.openfile('../path/to/datadir/example.gpkg')
        """
        _, ext = os.path.splitext(filepath)
        if len(ext) == 0:
            raise ValueError('Extension of the file is empty.')
        kwargs_ = kwargs
        if 'provider' in kwargs:
            provider =  self._find_by_id(kwargs_['provider'])
            del kwargs_['provider']
        else:
            provider = self.find_by_extension(ext[1:])
        return provider.open(filepath, *args, **kwargs_)

    def open(self, definition: dict) -> DataObject:
        """Открывает данные по описанию.

        Формат описания объектов данных индивидуален для каждого провайдера данных,
        однако многие элементы используются для всех провайдеров данных.

        Args:
            definition: Описание объекта данных.

        Пример::

            # Пример открытия GPKG файла::
            definition = { 'src': '../path/to/datadir/example.gpkg', 
                           'dataobject': 'tablename',
                           'provider': 'SqliteDataProvider' }
            table = provider_manager.open(definition)

        Пример открытия таблицы базы данных::

            definition = {"src": "localhost",
                          "db": "sample",
                          "user": "postgres",
                          "password": "postgres",
                          "dataobject": "public.world",
                          "provider": "PgDataProvider"}
            table = provider_manager.open(definition)

        """
        source = GenericSource(definition)
        return source.open()

    def createfile(self, filepath: str, schema, *args, **kwargs) -> DataObject:
        """Создает таблицу.

        :meth:`create` выполняет ту же функцию, но в более обобщенном виде.

        Args:
            filepath: Путь к создаваемой таблице.
            schema: Схема таблицы.
        """
        _, ext = os.path.splitext(filepath)
        if len(ext) == 0:
            raise ValueError('Extension of the file is empty.')
        provider = self.find_by_extension(ext[1:])
        return provider.create(filepath, schema, *args, **kwargs)

    def create(self, definition: dict) -> DataObject:
        """Создает и открывает данные из описания.

        Args:
            definition: Описание объекта данных.

        Псевдоним :meth:`create_open`.
        """
        return self.create_open(definition)

    def create_open(self, definition: dict) -> DataObject:
        """Создает и открывает данные из описания.

        Возможные параметры:
            - ``src`` - Строка, определяющая местоположение источника данных. Это может быть либо путь к файлу с расширением TAB, либо пустая строка (для таблицы, размещаемой в памяти).
            - ``schema`` - Схема таблицы. Задается массивом объектов, содержащих атрибуты.
            - ``hidden`` - Если указано True, то созданный объект не будет зарегистрирован в каталоге. См. также :meth:`open_hidden`

        Args:
            definition: Описание объекта данных.

        Пример::

            definition = {
                'src': '../path/to/datadir/edit/table.tab',
                'schema': attr.schema(
                    attr.string('field1'),
                    attr.integer('field2'),
                ),
            }
            table = provider_manager.create(definition)
        """
        schema = definition.pop('schema')
        dest = GenericDestination(schema, definition)
        return dest.create_open()

    def loaded_providers(self) -> dict:
        """Возвращает список всех загруженных провайдеров данных.

        Returns:
            Провайдеры в виде пар ``(Идентификатор : Описание)``.
        """
        return opener_instance.loaded_providers()

    def query(self, query_text: str, *tables) -> Table:
        """Выполняет SQL-запрос к перечисленным таблицам.

        Warning:
            Используйте :meth:`axipy.da.DataManager.query`.

        Args:
            query_text: Текст запроса.
            *tables: Список таблиц, к которым выполняется запрос. 

        Returns:
            Таблица, если результатом запроса является таблица.

        Пример::

                query_text = "SELECT * FROM world, caps WHERE world.capital = caps.capital"
                joined = provider_manager.query(query_text, world, caps)
        """
        return opener_instance.query(query_text, *tables)

    def read_contents(self, definition: Union[dict, str]) -> List[str]:
        """Читает содержимое источника данных.

        Обычно используется для источников, способных содержать несколько
        объектов данных.

        Args:
            definition: Описание источника данных.

        Returns:
            Имена объектов данных.

        Пример::

            >>> provider_manager.read_contents('../path/to/datadir/example.gpkg')
            ['world', 'worldcap']

            >>> world = provider_manager.openfile('../path/to/datadir/example.gpkg',
            ...         dataobject='world')
        """
        return opener_instance.read_contents(definition)
