from axipy.cpp_cs import ShadowCoordSystem, ShadowCoordinateTransformer
from .CoordSystem import CoordSystem, _internal_transform
from axipy.utl import Pnt, Rect
from typing import Optional, Union, List

from PySide2.QtCore import QRectF, QPointF


class CoordTransformer:
    """Класс для преобразования координат из одной СК в другую.
    При создании объекта трансформации в него передается исходная и целевая СК. После этого данный объект может использоваться 
    для преобразования данных между этими СК.

    Args:
        cs_from: Исходная СК.
        cs_to: Целевая СК.

    Пример::

        # Пример преобразования точки
        from axipy import *

        csLL = CoordSystem.from_prj("1, 104")
        csMercator = CoordSystem.from_prj("10, 104, 7, 0")
        inPoint = Pnt(10, 10)
        transformer = CoordTransformer(csLL, csMercator)
        outPoint = transformer.transform(inPoint)
        print('Result point:', outPoint)
        outRect = transformer.transform(Rect(0,0,10,10))
        print('Result rect:', outRect)
    """

    def __init__(self, cs_from: Union[CoordSystem, str], cs_to: Union[CoordSystem, str]):
        if isinstance(cs_from, str):
            cs_from = CoordSystem.from_string(cs_from)
        if isinstance(cs_to, str):
            cs_to = CoordSystem.from_string(cs_to)
        if not isinstance(cs_from, CoordSystem):
            raise Exception("Invalid parameter cs_from")
        if not isinstance(cs_to, CoordSystem):
            raise Exception("Invalid parameter cs_to")
        self.shadow = ShadowCoordinateTransformer(cs_from.shadow, cs_to.shadow)

    def transform(self, value: Union[Pnt, List[Pnt], QPointF, QRectF, Rect, List[QPointF]]) -> Union[Pnt, Rect, List[Pnt]]:
        """Преобразовывает точки из исходной СК в целевую СК.

        Args:
            value: Входное значение. Может быть точкой, массивом точек :class:`axipy.utl.Pnt` или :class:`axipy.utl.Rect`.

        Returns:
            Выходное значение. Тип зависит от входного и аналогичен ему.

        Raises:
            RuntimeError: Ошибка выполнения преобразования.
        """
        return _internal_transform(value, self.shadow.transform)
