#!/usr/bin/python3
# -*- coding: utf-8 -*-

"""
Пример создания отчета и сохрание его в pdf (A4). Отчет создается на 4-х листах.
Перед запуском необходимо открыть карту world.
Просматривать полученный отчет лучше с включенным режимом отображения 2-х страниц
"""

# Импорт
from axioma.render import *
from axioma.cs import *
from axioma.core.geometry import *
from axioma.core.dp import *
from axioma.mapinfo import *
from axioma.gui import *

from PyQt5.QtCore import  *
from PyQt5.QtGui import *
from PyQt5.QtWidgets import *
from PyQt5.QtPrintSupport import QPrinter

cs_ne = CoordSysFactory.defaultCoordSysFactory().createFromPrj("CoordSys Nonearth Units \"m\"" )

# Создание элемента отчета типа геометрический объект
# Текст рассматиривается как геометрия. Пример создания см. style.py
def createGeometryItem(report):
    print('Геометрия...')
    poly = QPolygonF()
    poly << QPointF(20,20) << QPointF(80,50) << QPointF(120,20) << QPointF(320,320) << QPointF(120,180) << QPointF(20,20)
    geom = Polygon(cs_ne, poly) # Задание геометрии
    style = MapBasicStyle().styleFromString("Pen (1, 5, 16711935) Brush (8, 255, 16777215)") # Задание стиля оформления
    geomItem = GeometryReportItem(report) # Элемент отчета
    geomItem.setGeometry(geom) # Геометрия элемента в координатах отчета
    geomItem.setStyle(style) # Стиль оформления геометрии
    report.addItem(geomItem) # Добавляем в отчет

# Создание элемента отчета типа растр из URL. Для локального файла использовать FileDefinition вместо WebDefinition
def createRasterItem(report):
    print('Растр...')
    dp = axioma.app.gui.dataProviderById("GdalDataProvider") # находим необходимый провайдер
    url = "https://upload.wikimedia.org/wikipedia/en/7/72/World_Map_WSF.svg.png" # Исходный растр
    if dp is not None:
        try:
            definition = WebDefinition(QUrl(url))
            rds = dp.openDataSource(definition) # Создаем источник данных
            obj = rds.openDataObject() # Создаем объект типа растр
            if (isinstance(obj, Raster)):
                lay = RasterLayer(obj) # Создаем растровый слой на базе существующего объекта
                rraster = QRectF(130, 0, 210, 145)
                rasterItem = RasterReportItem(lay, rraster, report)
                report.addItem(rasterItem) # Добавляем в отчет
            else:
                print('Объект не является растром. Что то пошло не так')
        except Exception as ex:
            print(ex)
            
# Создание элемента отчета типа карта. Берем первую из открытых
# Так же возможно создать элемент отчета не на базе MapView, создав временную карту (см. renderer.py)
def createMapItem(report):
    print('Карта...')
    if axioma.app.mainWindow.mapViewManager().mapViewCount():
        try:
            mapview =  axioma.app.mainWindow.mapViewManager().currentMapView() # Берем текущую открытую карту в системе
            r_map = mapview.viewport().sceneRect()
            w = 350;
            r_item = QRectF(2, 150, w, w * r_map.height() / r_map.width())
            viewport = MapViewport(r_item, r_map, mapview.viewport().coordSystem())
            
            #находим слой
            lay = mapview.map().rootLayerGroup().findChildByName("world")
            if lay is not None:
                props = lay.featureLayerProperties() #Свойства слоя
                props.setLabelingAttributeName("Страна")   # устанавливаем выражение для метки
                props.setShowLabels(True) # Показ меток
                props.setCentroidsVisible(True) # Показ центроидов
                
                # Сделаем так, чтобы метки не перекрывали друг друга
                policy = props.labelPlacementPolicy()
                policy.labelingType = LabelPlacementPolicy.DisallowOverlap
                props.setLabelPlacementPolicy(policy)

                lay.setProperties(props)
            
            mapItem = MapReportItem(mapview.map(), viewport, report)
            report.addItem(mapItem) # Добавляем в отчет
            print('Масштабная линейка...')
# Создание масштабной линейки
            ws = 30
            bar_rect = QRectF(r_item.right() - ws, r_item.bottom(), ws,15)
            scaleBarReportItem = ScaleBarReportItem(mapview.map(), mapItem.viewport(), bar_rect, report)
            report.addItem(scaleBarReportItem) # Добавляем в отчет
        except Exception as ex:
            print(ex)

# Создание элемента отчета типа таблица просмотра. Берем первую из открытых
def createBrowserItem(report):
    print('Таблица просмотра...')
    tableViewList = axioma.app.gui.widgetManager.allWidgets(TableWindow.staticMetaObject)
    if len(tableViewList):
        try:
            tableWindow = tableViewList[0] # Берем
            r_browser = QRectF(0, 380, 180, 100) # Прямоугольник элемента отчета
            browserItem = BrowserReportItem(tableWindow.browser(), r_browser, report)
            report.addItem(browserItem) # Добавляем в отчет
        except Exception as ex:
            print(ex)

# Создание элемента отчета, основанного на таблице
def createTableItem(report):
    print('Таблица...')
    if len(axioma.app.mainWindow.dataCatalog().allDataObjects()):
        obj = axioma.app.mainWindow.dataCatalog().allDataObjects()[0]
        if isinstance(obj, Table):
            try:
                r_table = QRectF(0, 480, 180, 200) # Прямоугольник элемента отчета
                tableItem = TableReportItem(obj, r_table, report)
                range = PageRange(2, 10) # Фильтр по строкам
                tableItem.setRange(range)
                fields = ['Страна', 'Столица', 'Население'] #Выбранные поля
                tableItem.setSelectedColumns(fields)
                tableItem.refreshValues()
                report.addItem(tableItem) # Добавляем в отчет
            except Exception as ex:
                print(ex)

# Создание элемента отчета типа Легенда. Берем первую из открытых карт и для первого слоя создаем легенду (см. legend,py)
# Так же возможно создать элемент отчета не на базе MapView, а создав временный слой на основе открытой таблицы (см. createLayerForDataObject в renderer.py)
def createLegendItem(report):
    print('Легенда...')
    if axioma.app.mainWindow.mapViewManager().mapViewCount():
        try:
            mapview = axioma.app.mainWindow.mapViewManager().currentMapView() # Получение текущей карты
            if mapview is not None:
                layers = mapview.map().flatLayerList() # Список слоёв карты
                for lay in layers:
                    if isinstance(lay, HasStyles): # Если есть легенда
                        legendItem = FeatureLegendItem(lay, lay.name(), report) # Создание легенды для векторного слоя
                        itemPosition = LegendItemPosition(report, legendItem, legendItem.basePoint())
                        r_legend = QRectF(210, 380, 50, 50)
                        legendReportItem = LegendReportItem(itemPosition, r_legend, report) # Создание элемента отчета
                        report.addItem(legendReportItem) # Добавляем в отчет
        except Exception as ex:
            print(ex)

# Создание контекста
def createContext(painter):
    deviceRect = sceneRect = QRectF(0,0,100,100)
    viewport = Viewport (deviceRect,sceneRect)
    return Context(painter, viewport)

def dumpReport(printer):
    try:
        report = Report(axioma.app.mainWindow)
        report.setHorisontalPagesCount(2)
        report.setVerticalPageCount(2)
        printer.setPaperSize(QPrinter.A4)
        report.setOptions(printer)
        printer.setOutputFormat(QPrinter.PdfFormat) # Pdf как выходной файл
        filename = QDir.temp().filePath('layout_out.pdf') # Имя выходного файла
        printer.setOutputFileName(filename)
        painter = QPainter(printer)
        context = createContext(painter)
        createGeometryItem(report)
        createRasterItem(report)
        createMapItem(report)
        createTableItem(report)
        createBrowserItem(report) # Устарело
        createLegendItem(report)
        report.render(context) # Отрисовка
        print(QObject().tr("Элементов отчета:"), report.count())
        print('Сохранено в файле ', filename)
        painter = None # Без этого возможно падение
    except Exception as ex:
        print(ex)
      

#printer = QPrinter(QPrinter.HighResolution) # Создание принтера с высоким разрешением (1200dpi для Pdf или зависит от принтера)
printer = QPrinter() #Создание принтера с обычным разрешением
dumpReport(printer)
