from axipy.cpp_gui import ShadowMapTool as ShadowTool
from axipy.cpp_gui import ShadowToolFactory as CppToolFactory
from axipy.da import DefaultKeys
from .ViewWrapper import View, MapView
from PySide2.QtCore import QEvent, QPointF, QPoint, QRectF, QRect
from PySide2.QtGui import (QMouseEvent, QKeyEvent, QWheelEvent, QPolygonF,
        QPaintEvent, QPainter)
from typing import Union, Optional
from axipy.utl import Rect, Pnt
from typing import Optional


class MapTool(ShadowTool):
    """Инструмент окна карты.

    При создании своего инструмента новый инструмент наследуется от этого класса,
    и переопределяет необходимые обработчики событий.

    Attributes:
        PassEvent (bool): Передать событие дальше. Значение :data:`False`.
        BlockEvent (bool): Прекратить обработку события. Значение :data:`True`.
        enable_on (Union[str, DefaultKeys]): Идентификатор наблюдателя для определения доступности инструмента. По умолчанию отсутствует.

    Пример::

        MyTool(MapTool):

            def mousePressEvent(self, event):
                print('mouse pressed')
                return self.PassEvent

    See also:
        :class:`axipy.da.StateManager`.
    """

    PassEvent = False
    BlockEvent = True
    enable_on = ''

    @property
    def view(self) -> MapView:
        """Отображение данных в окне."""
        return View._wrap(self.shadow_view())

    def to_scene(self, device: Union[QPoint, QRect]) -> Union[Pnt, Rect]:
        """Переводит точки из координат окна(пикселей) в координаты на карте.

        Args:
            device: Точки в координатах окна.

        Return:
            Точки в координатах карты.
        """
        if type(device) is QPoint:
            device = QPointF(device.x(), device.y())
            return Pnt._point_value_from_qt(self.view.device_to_scene_transform.map(device))
        if type(device) is QRect:
            device = QRectF(device.x(), device.y(), device.width(), device.height())
            return Rect._rect_value_from_qt(self.view.device_to_scene_transform.mapRect(device))
        raise TypeError('Unsuported type.')

    def to_device(self, scene: Union[Pnt, Rect]) -> Union[QPoint, QRect]:
        """Переводит точки из координат на карте в координаты окна(пиксели).

        Args:
            scene: Точки в координатах карты.

        Return:
            Точки в координатах окна.
        """
        if type(scene) is Pnt:
            mapped = self.view.scene_to_device_transform.map(Pnt._point_value_to_qt(scene))
            return mapped.toPoint()
        if type(scene) is Rect:
            mapped = self.view.scene_to_device_transform.mapRect(Rect._rect_value_to_qt(scene))
            return mapped.toRect()
        raise TypeError('Unsuported type.')

    def get_select_rect(self, device: QPoint, size: int = 3) -> Rect:
        """Возвращает прямоугольник в координатах карты для точки на экране.
        
        Удобно для использования при поиске объектов.
        
        Args:
            device: Точка в координатах окна.
            size: Размер квадрата в пикселях.
        
        Return:
            Прямоугольник в координатах карты.
        
        Пример::
        
            device_point = event.pos()
            bbox = self.get_select_rect(device_point, 30)
            features = table.items(bbox=bbox)
        """
        device_box = QRect(0, 0, size, size)
        device_box.moveCenter(device)
        return self.to_scene(device_box)

    def redraw(self):
        """Перерисовывает окно карты.
        
        Создает событие :class:`PySide2.QtGui.QPaintEvent` и помещает его в
        очередь обработки событий. Аналогично
        :meth:`PySide2.QtWidgets.QWidget.update`.
        """
        self.view.widget.update()

    def is_snapped(self) -> bool:
        """Проверяет, сработала ли привязка к элементам карты или отчета
        для текущего положения указателя мыши.

        See also:
            :meth:`snap`, :meth:`snap_device`.
        """
        return self.isSnapped()

    def snap(self, default_value: Pnt = None) -> Optional[Pnt]:
        """Возвращает исправленные координаты, если сработала привязка к
        элементам в единицах измерения карты или отчета.
        
        Args:
            default_value: Значение по умолчанию.
        
        Возвращает значение по умолчанию, если не сработала привязка к элементам
        карты или отчета.
        
        Пример::
        
            point = self.to_scene(event.pos())
            current_point = self.snap(point)
        
        See also:
            :meth:`is_snapped`, :meth:`snap_device`, :meth:`to_scene`.
        """
        if not self.is_snapped():
            return default_value
        return Pnt._point_value_from_qt(self.snapPoint())

    def snap_device(self, default_value: QPoint = None) -> Optional[QPoint]:
        """Возвращает исправленные координаты, если сработала привязка к
        элементам в единицах измерения окна карты (виджета).
        
        Args:
            default_value: Значение по умолчанию.
        
        Возвращает значение по умолчанию, если не сработала привязка к элементам
        карты или отчета.
        
        Пример::
        
            device_point = event.pos()
            current_device_point = self.snap_device(device_point)

        See also:
            :meth:`is_snapped`, :meth:`snap`.
        """
        if not self.is_snapped():
            return default_value
        return self.to_device(self.snap())

    def handleEvent_impl(self, event: QEvent) -> bool:
        result = self.handleEvent(event)
        if result == self.BlockEvent:
            return self.BlockEvent
        return super().handleEvent_impl(event)

    def handleEvent(self, event: QEvent) -> Optional[bool]:
        """Первичный обработчик всех событий инструмента.
        
        Если событие не блокируется этим обработчиком, то оно будет передано
        дальше в соответствующий специализированный обработчик
        :meth:`mousePressEvent`, :meth:`keyReleaseEvent` и прочие в зависимости
        от типа.
        
        Args:
            event: Событие.
        
        Return:
            :attr:`BlockEvent`, чтобы блокировать дальнейшую обработку события.
        """
        pass

    def paintEvent(self, event: QPaintEvent, painter: QPainter):
        """Обрабатывает событие отрисовки.

        Args:
            event: Событие отрисовки.
            painter: QPainter для рисования поверх виджета
        """
        pass

    def mousePressEvent_impl(self, event: QEvent) -> bool:
        result = self.mousePressEvent(event)
        if result != self.PassEvent:
            return self.BlockEvent
        return super().mousePressEvent_impl(event)

    def mouseReleaseEvent_impl(self, event: QEvent) -> bool:
        result = self.mouseReleaseEvent(event)
        if result != self.PassEvent:
            return self.BlockEvent
        return super().mouseReleaseEvent_impl(event)

    def mouseDoubleClickEvent_impl(self, event: QEvent) -> bool:
        result = self.mouseDoubleClickEvent(event)
        if result != self.PassEvent:
            return self.BlockEvent
        return super().mouseDoubleClickEvent_impl(event)

    def mouseMoveEvent_impl(self, event: QEvent) -> bool:
        result = self.mouseMoveEvent(event)
        if result == self.BlockEvent:
            return self.BlockEvent
        return super().mouseMoveEvent_impl(event)

    def keyPressEvent_impl(self, event: QEvent) -> bool:
        result = self.keyPressEvent(event)
        if result != self.PassEvent:
            return self.BlockEvent
        return super().keyPressEvent_impl(event)

    def keyReleaseEvent_impl(self, event: QEvent) -> bool:
        result = self.keyReleaseEvent(event)
        if result != self.PassEvent:
            return self.BlockEvent
        return super().keyReleaseEvent_impl(event)

    def wheelEvent_impl(self, event: QEvent) -> bool:
        result = self.wheelEvent(event)
        if result != self.PassEvent:
            return self.BlockEvent
        return super().wheelEvent_impl(event)

    def mousePressEvent(self, event: QMouseEvent) -> Optional[bool]:
        """Обрабатывает событие нажатия клавиши мыши.

        Args:
            event: Событие нажатия клавиши мыши.

        Return:
            :attr:`PassEvent`, чтобы пропустить событие дальше по цепочке
            обработчиков. :data:`None` или :attr:`BlockEvent`, чтобы блокировать дальнейшую
            обработку события.
        """
        return self.PassEvent

    def mouseReleaseEvent(self, event: QMouseEvent) -> Optional[bool]:
        """Обрабатывает событие отпускания клавиши мыши.

        Args:
            event: Событие отпускания клавиши мыши.

        Return:
            :attr:`PassEvent`, чтобы пропустить событие дальше по цепочке
            обработчиков. :data:`None` или :attr:`BlockEvent`, чтобы блокировать дальнейшую
            обработку события.
        """
        return self.PassEvent

    def mouseDoubleClickEvent(self, event: QMouseEvent) -> Optional[bool]:
        """Обрабатывает событие двойного клика мыши.

        Args:
            event: Событие двойного клика мыши.

        Return:
            :attr:`PassEvent`, чтобы пропустить событие дальше по цепочке
            обработчиков. :data:`None` или :attr:`BlockEvent`, чтобы блокировать дальнейшую
            обработку события.
        """
        return self.PassEvent

    def mouseMoveEvent(self, event: QMouseEvent) -> Optional[bool]:
        """Обрабатывает событие перемещения мыши.

        Args:
            event: Событие перемещения мыши.

        Return:
            :attr:`PassEvent` или :data:`None`, чтобы пропустить событие дальше по цепочке
            обработчиков. :attr:`BlockEvent`, чтобы блокировать дальнейшую
            обработку события.
        """
        return self.PassEvent

    def keyPressEvent(self, event: QKeyEvent) -> Optional[bool]:
        """Обрабатывает событие нажатия клавиши клавиатуры.

        Args:
            event: Событие нажатия клавиши клавиатуры.

        Return:
            :attr:`PassEvent`, чтобы пропустить событие дальше по цепочке
            обработчиков. :data:`None` или :attr:`BlockEvent`, чтобы блокировать дальнейшую
            обработку события.
        """
        return self.PassEvent

    def keyReleaseEvent(self, event: QKeyEvent) -> Optional[bool]:
        """Обрабатывает событие отпускания клавиши клавиатуры.

        Args:
            event: Событие отпускания клавиши клавиатуры.

        Return:
            :attr:`PassEvent`, чтобы пропустить событие дальше по цепочке
            обработчиков. :data:`None` или :attr:`BlockEvent`, чтобы блокировать дальнейшую
            обработку события.
        """
        return self.PassEvent

    def wheelEvent(self, event: QWheelEvent) -> Optional[bool]:
        """Обрабатывает событие колеса мыши.

        Args:
            event: Событие колеса мыши.

        Return:
            :attr:`PassEvent`, чтобы пропустить событие дальше по цепочке
            обработчиков. :data:`None` или :attr:`BlockEvent`, чтобы блокировать дальнейшую
            обработку события.
        """
        return self.PassEvent

    def deactivate(self):
        """Выполняет действия непосредственно перед выключением инструмента и
        перед его удалением.
        
        Переопределите этот метод, чтобы задать свои действия.
        """
        pass

    def deactivate_impl(self):
        self.deactivate()

    @staticmethod
    def reset():
        """Переключает текущий инструмент на инструмент по умолчанию.
        
        Обычно инструментом по умолчанию является Выбор.
        """
        CppToolFactory.requestReset()
