from .data_provider import DataProvider, Table
from .source import Source, Destination, Schema
from typing import List, Optional


class PostgreSource(Source):
    pass


class PostgreDestination(Destination):
    pass


class PostgreDataProvider(DataProvider):
    """Провайдер для Базы Данных PostgreSQL."""
    _identifier = 'PgDataProvider'

    def get_source(self, host: str, db_name: str, user: str,
                   password: str, port: int = 5432, dataobject: Optional[str] = None, sql: Optional[str] = None) -> Source:
        """Создает описательную структуру для источника данных. Она в дальнейшем может быть использована при открытии данных :meth:`ProviderManager.open`.

        Note:
            В качестве таблицы можно указать либо ее наименование `dataobject` либо текст запроса `sql`.

        Note:
            Ссылку на провайдер можно получить через глобальную переменную :attr:`axipy.provider_manager.postgre`.

        Args:
            host: Адрес сервера.
            db_name: Имя базы данных.
            user: Имя пользователя.
            password: Пароль.
            port: Порт.
            dataobject: Имя таблицы.
            sql: SQL-запрос. Если указан, то он имеет более высокий приоритет по отношению к значению `dataobject`.

        Пример с указанием имени таблицы::

            definition = provider_manager.postgre.get_source('localhost', 'test', 'postgres', 'postgres', dataobject='world')
            table = provider_manager.open(definition)

        Пример с указанием текста запроса::

            definition = provider_manager.postgre.get_source('localhost', 'test', 'postgres', 'postgres', sql="select * from world where Страна like 'Р%'")
            table = provider_manager.open(definition)
        """
        return PostgreSource(
            Source._provider(self.id),
            {
                'host': host,
                'port': port,
                'db': db_name,
                'user': user,
                'password': password,
                'dataobject': dataobject,
                'sql' : sql
            }
        )

    def get_destination(self, schema: Schema, dataobject: str, db_name: str,
                        host: str, user: str, password: str, port: int = 5432
                        ) -> Destination:
        """Создает назначение объекта данных.

        Args:
            schema: Схема таблицы.
            dataobject: Имя таблицы.
            db_name: Имя базы данных.
            host: Адрес сервера.
            user: Имя пользователя.
            password: Пароль.
            port: Порт.
        """
        return PostgreDestination(schema,
                                  Source._provider(self.id),
                                  {
                                      'src': host,
                                      'port': port,
                                      'db': db_name,
                                      'user': user,
                                      'password': password,
                                      'dataobject': dataobject
                                  }
                                  )
