from PySide2.QtWidgets import QDialog, QPushButton, QWidget
from PySide2.QtGui import QResizeEvent,  QPaintEvent, QPainter
from axipy.cpp_gui import ShadowChooseCoordSystemWidget, ShadowStyleButton
from axipy.cs import CoordSystem
from axipy.da import Style
from PySide2.QtCore import QObject, Signal


class ChooseCoordSystemDialog(QDialog):
    """Диалог выбора координатной системы."""

    def __init__(self, coordsystem: CoordSystem):
        """Конструктор.

        Args:
            coordsystem: Система координат по умолчанию.

        Пример::

            csMercator = CoordSystem.from_prj("10, 104, 7, 0")
            dialog = ChooseCoordSystemDialog(csMercator)
            if dialog.exec() == QDialog.Accepted:
                result_cs = dialog.chosenCoordSystem()
                print("Выбрано:", result_cs.description)
        """
        super().__init__()
        if self.layout() is not None:
            self.layout().setContentsMargins(0, 0, 0, 0)

        self.shadow = ShadowChooseCoordSystemWidget(coordsystem.shadow, self)

    def chosenCoordSystem(self) -> CoordSystem:
        """Возвращает выбранную в диалоге систему координат."""
        return CoordSystem._wrap(self.shadow.chosenCoordSystem())


class StyledButton(QPushButton):
    """Кнопка, отображающая стиль и позволяющая его менять

    Note:
        Сигнал `styleChanged` испускается при смене стиля.

    Пример добавления кнопки на диалог::

        style = Style.from_mapinfo("Pen (1, 2, 8421504)  Brush (2, 255, 0)")

        class Dialog(QDialog):
            def __init__(self, parent = None):
                QDialog.__init__(self, parent)
                self.pb = StyledButton( style, self)
                self.pb.styleChanged.connect(self.styleResult)
                self.pb.setGeometry(100, 100, 100, 50)

            def styleResult(self):
                print('Стиль изменен', self.pb.style())

        dialog = Dialog()
        dialog.exec()
    """
    styleChanged = Signal()

    def __init__(self, style: Style, parent: QWidget = None):
        """Конструктор.

        Args:
            style: Стиль по умолчанию.
        """
        super().__init__(parent)
        self.shadow = ShadowStyleButton(self, style.shadow)
        self.shadow.styleChanged.connect(self.styleChanged)

    def style(self) -> Style:
        """Результирующий стиль."""
        return Style._wrap(self.shadow.get_style())

    def paintEvent(self, event: QPaintEvent) -> None:
        super().paintEvent(event)
        painter = QPainter(self)
        painter.drawPixmap(2, 2, self.shadow.pixmap())

    def resizeEvent(self, event: QResizeEvent) -> None:
        super().resizeEvent(event)
        self.shadow.reinitPixmap()
