
from axipy.da import ValueObserver, state_manager, Table, data_manager, Geometry
from axipy.sql import get_database
from axipy import Rect
from axipy.gui import MapView, selection_manager, view_manager
from typing import List, Union
from PySide2.QtCore import QLocale
from PySide2.QtSql import QSqlQuery
from axipy.concurrent import task_manager


def quick_envelope(geometries: List[Geometry]) -> Rect:
    """
    Функция возвращает нормализованный ограничивающий прямоугольник.
    """
    if len(geometries) == 0:
        return None
    target = geometries[0].bounds
    target.normalize()
    for geom in geometries:
        r = geom.bounds
        r.normalize()
        target.xmin = min(target.xmin, r.xmin)
        target.xmax = max(target.xmax, r.xmax)
        target.ymin = min(target.ymin, r.ymin)
        target.ymax = max(target.ymax, r.ymax)
    return target


class DataManagerGuard:
    """
    В конструкторе добавляем в DataManager, а в деструкторе удаляем.
    TODO: Правильный путь - это дать возможность добавлять таблицы, которые
    не отображаются в открытых данных. Очень удобно для временных таблиц
    """

    def __init__(self, table) -> None:
        from secrets import token_hex
        self._table = table
        self._table.name = 'selection_' + token_hex(nbytes=16)
        data_manager.add(self._table)

    def __del__(self):
        data_manager.remove(self._table)

    @property
    def data(self) -> Table:
        return self._table

    @property
    def name(self) -> str:
        return self._table.name


def wrap_tmp_table(table: Table):
    return DataManagerGuard(table)


def selection_as_tmp_table() -> DataManagerGuard:
    """
    Создаём временную таблицу на основе выборке, которую регистрируем в DataManager,
    чтобы её можно было использовать в SQL запросах
    """
    table = selection_manager.get_as_table()
    return wrap_tmp_table(table)


def current_editable_table() -> Table:
    """
    Функция возвращает таблицу для текущего редактируемого слоя активной карты
    """
    map_view = view_manager.active  # type: MapView
    if not isinstance(map_view, MapView):
        return None
    editable_layer = map_view.editable_layer
    if editable_layer is None:
        return None
    editable_table = editable_layer.data_object
    if not isinstance(editable_table, Table):
        print(
            f"Типом источника данных должен быть axipy.da.Table. Вместо этого {editable_table}")
        return None
    return editable_table


def ensure_editable_table() -> Table:
    table = current_editable_table()
    assert table is not None
    return table


def normalize_if_not_valid(
        geometries: Union[List[Geometry], Geometry]) -> Union[List[Geometry], Geometry]:
    def normalize(geometries: List[Geometry]):
        res = []
        for g in geometries:
            if not g.is_valid:
                res.append(g.normalize())
            else:
                res.append(g)
        return res
    if isinstance(geometries, list):
        return normalize(geometries)
    else:
        return normalize([geometries])[0]


def text_to_float(text: str, locale: QLocale) -> float:
    """
    Преобразовывает текстовою строку в float с использованием локали.

    RuntimeError при ошибках преобразования
    """
    value, ok = locale.toFloat(text)
    if not ok:
        raise RuntimeError("Не удалось преобразовать значение "
                           "поля ввода в число.")
    return value

def ensure_sql_init_for_bg_thread():
    query = QSqlQuery(get_database())
    task_manager.run_in_gui(lambda: query.exec_("Select 'Init SQL Engine'"))