"""Модуль меню главного окна ГИС Аксиома."""

import inspect
import traceback
import uuid
from pathlib import Path
from typing import Any, Callable, List, Optional, Union, cast

import axipy
from axipy._internal._decorator import _back_compat_params, _deprecated_by
from axipy._internal._shadow_instance_factory import _shadow_manager
from axipy.cpp_gui import (
    ShadowButtonPositionIndex,
    ShadowMapTool,
    ShadowToolFactory,
    ShadowViewType,
)
from axipy.da import Observer, ObserverManager
from axipy.gui import MapTool, ViewTool
from PySide2.QtGui import QIcon
from PySide2.QtWidgets import QAction

__all__: List[str] = [
    "Button",
    "ActionButton",
    "ToggleButton",
    "ToolButton",
    "Position",
    "Separator",
    "SystemActionButton",
]


class Button:
    """
    Кнопка с инструментом для добавления в меню. Абстрактный класс.

    Для создания объекта этого класса используйте
    :class:`ActionButton`, :class:`ToolButton`, :class:`SystemActionButton`.
    """

    def __init__(self, action: QAction, observer: Optional[Union[str, Observer]] = None) -> None:
        if type(self) is Button:
            raise NotImplementedError

        self._action: QAction = action

        observer_typed: Optional[axipy.Observer]
        name: Optional[str] = ObserverManager._to_name(observer)
        if name is None:
            observer_typed = None
        else:
            observer_typed = ObserverManager().get(name, None)
        self._observer: Optional[axipy.Observer] = observer_typed

    @staticmethod
    def _title_to_id(title: str) -> str:
        # Creates identifier from title.
        table = str.maketrans(
            "абвгдеёжзийклмнопрстуфхцчшщъыьэюяАБВГДЕЁЖЗИЙКЛМНОПРСТУФХЦЧШЩЪЫЬЭЮЯ ",
            "abvgdeejzijklmnoprstufhzcss_y_euaABVGDEEJZIJKLMNOPRSTUFHZCSS_Y_EUA_",
        )
        return title.translate(table)

    def _init_action(
        self,
        title: str,
        icon: Union[str, Path, QIcon] = "",
        tooltip: Optional[str] = None,
        doc_file: Optional[Union[str, Path]] = None,
    ) -> None:
        self.action.setText(title)

        if isinstance(icon, str):
            icon = QIcon(icon)
        elif isinstance(icon, Path):
            icon = QIcon(str(icon))
        self.action.setIcon(icon)

        if tooltip:
            self.action.setToolTip(tooltip)

        self.__set_doc_file(doc_file)

        self.action.setObjectName(self._title_to_id(title))

    def __set_doc_file(self, doc_file: Optional[Union[str, Path]]) -> None:
        if doc_file is not None:
            self.action.setWhatsThis(self.__full_doc_file(doc_file))

    def __full_doc_file(self, path: Union[str, Path]) -> str:
        doc_path: str = path if isinstance(path, str) else str(path)
        return "file:///{}".format(doc_path.replace("\\", "/"))

    @property
    def action(self) -> QAction:
        """
        Возвращает ссылку на объект :class:`PySide2.QtWidgets.QAction`. Через него можно
        производить дополнительные необходимые действия через объект `Qt`.

        Пример задания всплывающей подсказки, используя метод класса :class:`PySide2.QtWidgets.QAction`::

            button.action.setToolTip("Всплывающая подсказка")
        """
        return self._action

    @property
    def observer(self) -> Optional[Observer]:
        """Возвращает наблюдатель для определения доступности кнопки."""
        return self._observer

    @property
    def observer_id(self) -> str:
        """Возвращает идентификатор наблюдателя для определения доступности кнопки."""
        return self._observer.name if self._observer else ""

    def remove(self) -> None:
        """Удаляет кнопку из меню."""
        _shadow_manager.menu_bar.remove(self.action)


class Separator(Button):
    """Разделитель."""

    def __init__(self) -> None:
        """Конструктор класса."""
        separator = QAction()
        separator.setObjectName(uuid.uuid4().hex[:6].upper())
        separator.setSeparator(True)
        super().__init__(separator)


class SystemActionButton(Button):
    """
    Кнопка для действия, присутствующего в системе.

    See also:
        Примеры использования см. :class:`axipy.ActionManager`
    """

    def __init__(self, name: str) -> None:
        """
        Конструктор класса.

        Args:
            name: идентификатор действия
                Полный список доступных идентификаторов можно получить
                посредством :meth:`axipy.action_manager.keys()`.
        """
        action = axipy.ActionManager().get(name)
        if not action:
            raise ValueError(f"Action {name} not found.")
        super().__init__(action)

    def remove(self) -> None:
        _shadow_manager.menu_bar.removeSystem(self.action)


class ActionButton(Button):
    """
    Кнопка с действием.

    See also:
        :class:`axipy.ObserverManager`.

    .. literalinclude:: /../../tests/doc_examples/test_example_menubar.py
        :caption: Пример со встроенным наблюдателем.
        :pyobject: test_run_example_menubar_button
        :lines: 2-
        :dedent: 4
        :end-before: # finish

    .. literalinclude:: /../../tests/doc_examples/test_example_menubar.py
        :caption: Пример со пользовательским наблюдателем.
        :pyobject: test_run_example_menubar_button_observer
        :lines: 2-
        :dedent: 4
        :end-before: # finish
    """

    def __init__(
        self,
        title: str,
        on_click: Callable[[], Any],
        icon: Union[str, Path, QIcon] = "",
        enable_on: Optional[Observer] = None,
        tooltip: Optional[str] = None,
        doc_file: Optional[Union[str, Path]] = None,
    ) -> None:
        """
        Конструктор класса.

        Args:
            title: Текст.
            on_click: Действие на нажатие. Передается функция, которая будет вызвана при нажатии на кнопку.
            icon: Иконка. Может быть путем к файлу или адресом ресурса.
            enable_on: Наблюдатель, для определения доступности кнопки.
            tooltip: Строка с дополнительной короткой информацией по данному действию.
            doc_file: Полный путь к html файлу с документацией.
        """
        action = QAction()
        super().__init__(action, enable_on)
        self._init_action(title, icon, tooltip, doc_file)

        if on_click is not None:
            self.action.triggered.connect(on_click)


class ToggleButton(Button):
    """
    Переключаемая кнопка.
    Может находиться в двух состояниях: включено или выключено.
    """

    def __init__(
        self,
        title: str,
        on_toggle: Callable[[bool], Any],
        add_to_tool_group: bool = False,
        icon: Union[str, Path, QIcon] = "",
        enable_on: Optional[Observer] = None,
        tooltip: Optional[str] = None,
        doc_file: Optional[Union[str, Path]] = None,
    ) -> None:
        """
        Конструктор класса.

        Args:
            title: Текст.
            on_toggle: Действие на нажатие. Передается функция, которая будет вызвана при нажатии на кнопку.
            add_to_tool_group: Добавить кнопку в группу кнопок Аксиомы. В этом случае кнопка будет оставаться нажатой,
                только если нет других нажатых кнопок Аксиомы.
            icon: Иконка. Может быть путем к файлу или адресом ресурса.
            enable_on: Наблюдатель, для определения доступности кнопки.
            tooltip: Строка с дополнительной короткой информацией по данному действию.
            doc_file: Полный путь к html файлу с документацией.
        """
        action = QAction()
        super().__init__(action, enable_on)
        self._init_action(title, icon, tooltip, doc_file)

        self._add_to_tool_group: bool = add_to_tool_group
        action.setCheckable(True)
        action.toggled.connect(on_toggle)

    @property
    def is_toggled(self) -> bool:
        """Возвращает признак, нажата ли кнопка."""
        return self.action.isChecked()


class ToolButton(Button):
    """
    Переключаемая кнопка с инструментом.

    See also:
        :class:`axipy.ViewTool`.
        :class:`axipy.ObserverManager`.

    .. literalinclude:: /../../tests/doc_examples/test_example_menubar.py
        :caption: Пример
        :pyobject: test_run_example_menubar_tbutton
        :lines: 2-
        :dedent: 4
        :end-before: # finish
    """

    class _ToolFactory(ShadowToolFactory):

        def __init__(self, factory: Callable[[ViewTool.ViewType], Optional[ViewTool]]) -> None:
            super().__init__()
            self.__factory: Callable[[ViewTool.ViewType], Optional[ViewTool]] = factory
            self.__factory_has_param: bool = len(inspect.signature(self.__factory).parameters) == 1

        def __get_view_type(self, shadow_view_type: ShadowViewType) -> ViewTool.ViewType:
            if shadow_view_type == ShadowViewType.MAP_VIEW:
                return ViewTool.ViewType.MAP_VIEW
            elif shadow_view_type == ShadowViewType.REPORT_VIEW:
                return ViewTool.ViewType.REPORT_VIEW
            else:
                raise RuntimeError(f"Unknown ShadowViewType - {shadow_view_type}")

        def create(self, shadow_view_type: ShadowViewType) -> "ShadowMapTool":
            if self.__factory_has_param:
                tool = self.__factory(self.__get_view_type(shadow_view_type))
            else:
                # Either backwards compatibility (MapTool), or incorrect signature provided
                view_type = self.__get_view_type(shadow_view_type)
                if view_type != ViewTool.ViewType.MAP_VIEW:
                    return None  # type: ignore[return-value]
                tool = self.__factory()  # type: ignore[call-arg]
                if not isinstance(tool, axipy.MapTool):
                    raise TypeError(
                        "Incorrect parameter length in callable signature. Callable should take 1 positional argument."
                    )

            if tool is None:
                return None  # type: ignore[return-value]

            shadow_tool = tool._ViewTool__shadow  # type: ignore[attr-defined]
            shadow_tool.setParent(self)
            return shadow_tool

    @_back_compat_params({"on_click": "on_create_tool"})
    def __init__(
        self,
        title: str,
        on_create_tool: Callable[["axipy.ViewTool.ViewType"], Optional["axipy.ViewTool"]],
        icon: Union[str, Path, QIcon] = "",
        enable_on: Optional[Observer] = None,
        tooltip: Optional[str] = None,
        doc_file: Optional[Union[str, Path]] = None,
        on_toggle: Optional[Callable[[bool], None]] = None,
    ) -> None:
        """
        Конструктор класса.

        Args:
            title: Текст.
            on_create_tool: Функция создающая инструмент.
                Передается функция, которая будет вызвана в момент создания инструмента для карты.
                Функция должна принимать обязательный параметр типа :attr:`axipy.ViewTool.ViewType` и возвращать
                экземпляр :class:`axipy.ViewTool` или :obj:`None`.
            icon: Иконка. Может быть путем к файлу или адресом ресурса.
            enable_on: Наблюдатель, для определения доступности кнопки.
            tooltip: Строка с дополнительной короткой информацией по данному действию.
            doc_file: Полный путь к html файлу с документацией.
            on_toggle: Действие на нажатие. Передается функция, которая будет вызвана при нажатии на кнопку.
        """
        action = QAction()
        # backwards compatibility
        if inspect.isclass(on_create_tool) and issubclass(on_create_tool, MapTool):
            # TODO: refactor w/out type ignore
            enable_on = enable_on or on_create_tool.enable_on  # type: ignore[assignment]
        else:
            if not callable(on_create_tool):
                raise ValueError("Object is not callable")
            signature = inspect.signature(on_create_tool)
            if len(signature.parameters) > 1:  # backwards compatibility
                raise ValueError("Callable must must have 1 parameter")

        super().__init__(action, enable_on)
        self._init_action(title, icon, tooltip, doc_file)
        self.__factory = on_create_tool

        if on_toggle is not None:
            self.action.toggled.connect(on_toggle)

    def _create_tool_factory(self) -> "_ToolFactory":
        return self._ToolFactory(self.__factory)

    @property
    def is_toggled(self) -> bool:
        """Возвращает признак, нажата ли кнопка."""
        return self.action.isChecked()


class Position:
    """
    Положение кнопки в меню.

    Пример::

        import axipy

        pos = axipy.Position("Основные", "Команды")
    """

    def __init__(self, tab: str, group: str) -> None:
        """
        Конструктор класса.

        Args:
            tab: Название вкладки.
            group: Название группы.
        """
        self._tab = tab
        self._group = group

    def __find_tab_id(self, tab_name: str) -> Optional[str]:
        tabs = _shadow_manager.menu_bar.tabs()
        if tab_name not in tabs:
            for key, value in tabs.items():
                if tab_name == value:
                    return key
        return None

    def __find_group_id(self, name: str) -> Optional[str]:
        pairs = _shadow_manager.menu_bar.groups()
        if name not in pairs:
            for key, value in pairs.items():
                if name == value:
                    return key
        return None

    def __find_tab_id_titled(self) -> str:
        tab_id = self.__find_tab_id(self._tab)
        if tab_id is None:
            tab_id = Button._title_to_id(self._tab)
        return tab_id

    def __find_group_id_titled(self) -> str:
        group_id = self.__find_group_id(self._group)
        if group_id is None:
            group_id = Button._title_to_id(self._group)
        return group_id

    def __add_action_button(
        self, b: Union[ActionButton, SystemActionButton, Separator], position: List[str], size: int
    ) -> None:
        _shadow_manager.menu_bar.add_action_button(
            b.action,
            position,
            b.observer_id,
            size,
        )

    def __add_toggle_button(
        self,
        b: ToggleButton,
        position: List[str],
        size: int,
    ) -> None:
        _shadow_manager.menu_bar.add_toggle_button(
            b.action,
            position,
            b.observer_id,
            size,
            b._add_to_tool_group,
        )

    def __add_tool_button(self, b: ToolButton, position: List[str], size: int) -> None:
        _shadow_manager.menu_bar.add_tool_button(
            b.action,
            position,
            b._create_tool_factory(),
            b.observer_id,
            size,
        )

    def add(self, button: Button, size: int = 2) -> None:
        """
        Добавляет кнопку в текущее положение.

        Args:
            button: Кнопка.
            size: Размер кнопки. Маленькая кнопка - 1. Большая кнопка - 2.
        """
        tab_id: str = self.__find_tab_id_titled()
        group_id: str = self.__find_group_id_titled()

        position: List[str] = cast(List[str], [None for _ in range(ShadowButtonPositionIndex.COUNT)])
        position[ShadowButtonPositionIndex.TabId] = tab_id
        position[ShadowButtonPositionIndex.TabName] = self._tab
        position[ShadowButtonPositionIndex.GroupId] = group_id
        position[ShadowButtonPositionIndex.GroupName] = self._group

        try:
            if isinstance(button, (ActionButton, SystemActionButton, Separator)):
                self.__add_action_button(button, position, size)
            elif isinstance(button, ToggleButton):
                self.__add_toggle_button(button, position, size)
            elif isinstance(button, ToolButton):
                self.__add_tool_button(button, position, size)
            else:
                raise RuntimeError(f"Unknown Button type - {button}")
        except Exception:
            traceback.print_exc()


def _apply_deprecated() -> None:
    # Using getattr to hide deprecated objects from axipy namespace on IDE inspections
    getattr(__all__, "extend")(
        (
            "create_button",
            "remove",
            "get_position",
        )
    )

    def create_action(_cls: Button, title: str, icon: Union[str, QIcon] = "") -> QAction:
        if not isinstance(icon, QIcon):
            icon = QIcon(icon)
        action = QAction(icon, title)
        action.setObjectName(Button._title_to_id(title))
        return action

    setattr(Button, "create_action", classmethod(_deprecated_by()(create_action)))

    # noinspection PyShadowingNames
    def create_button(
        title: str,
        on_click: Union[Callable[[], Any], Callable[[], MapTool], MapTool],
        icon: Union[str, QIcon] = "",
        enable_on: Optional[Observer] = None,
    ) -> Button:
        """
        Warning:
            .. deprecated:: 3.5
                Используйте конструктор наследников :class:`axipy.menubar.Button`.
        """
        if inspect.isclass(on_click) and issubclass(on_click, MapTool):
            return ToolButton(title, on_click, icon, enable_on)
        return ActionButton(title, cast(Callable[..., Any], on_click), icon, enable_on)

    # noinspection PyShadowingNames
    def remove(button: Union[QAction, Button]) -> None:
        """
        Warning:
            .. deprecated:: 3.5
                Используйте метод :meth:`axipy.Button.remove`.
        """
        if isinstance(button, Button):
            button = button.action
        _shadow_manager.menu_bar.remove(button)

    # noinspection PyShadowingNames
    def get_position(tab: str, group: str) -> Position:
        """
        Warning:
            .. deprecated:: 3.5
                Используйте конструктор :class:`axipy.Position`.
        """
        return Position(tab, group)

    # noinspection PyShadowingNames
    def add(button: Union[QAction, Button], position: Position, size: int = 2) -> None:
        """
        Warning:
            .. deprecated:: 3.5
                Используйте метод :meth:`axipy.Position.add`.
        """
        if isinstance(button, QAction):
            button_fixed: Button = object.__new__(Button)
            button_fixed._action = button
            button_fixed._observer = None
            position.add(button_fixed, size)
            return None

        position.add(button, size)

    # noinspection PyShadowingNames
    def title_to_id(title: str) -> str:
        """
        Warning:
            .. deprecated:: 3.5
        """
        return Button._title_to_id(title)

    globals().update(
        create_button=_deprecated_by("axipy.ActionButton, axipy.ToolButton")(create_button),
        remove=_deprecated_by("axipy.Button.remove")(remove),
        get_position=_deprecated_by("axipy.Position")(get_position),
        add=_deprecated_by("axipy.Position.add")(add),
        title_to_id=_deprecated_by()(title_to_id),
    )


_apply_deprecated()
