from typing import TYPE_CHECKING, Iterator, List, Optional, cast

from axipy._internal._decorator import _experimental
from axipy._internal._shadow_instance_factory import _shadow_manager
from axipy._internal._utils import _AxiRepr, _AxiReprMeta, _Singleton
from axipy.da import Feature, Table
from PySide2.QtCore import Signal

if TYPE_CHECKING:
    from axipy.cpp_gui import ShadowSelectionManager

__all__: List[str] = [
    "_TargetObjectManager",
]


class __TargetObjectManager(_Singleton, _AxiRepr, metaclass=_AxiReprMeta): ...


class _TargetObjectManager(__TargetObjectManager):
    """
    Класс доступа к отмеченным для изменения в окне карты объектам.

    Note:
        Создание :class:`axipy._TargetObjectManager` не требуется,
        используйте объект :attr:`axipy._target_object_manager`.
    """

    @property
    def table(self) -> Optional[Table]:
        """
        Возвращает таблицу, являющуюся источником отмеченных объектов.

        Если ничего не отмечено, то None.
        """
        return cast(Optional[Table], Table._wrap(self._shadow.target_table()))

    def from_selection(self) -> None:
        """Устанавливает текущую выборку как объекты, помеченные для изменения."""
        self._shadow.selectionToTarget()

    @property
    def ids(self) -> List[int]:
        """Возвращает список идентификаторов помеченных записей."""
        return self._shadow.target_ids()

    def clear(self) -> None:
        """Убирает отметку."""
        self._shadow.clearTarget()

    def get_as_cursor(self) -> Iterator[Feature]:
        """
        Возвращает помеченные записи в виде итератора по записям.

        Пример::

            for f in axipy._target_object_manager.get_as_cursor():
                print('Feature id={}. Страна={}'.format(f.id, f['Страна']))
        """
        if self.table is None:
            return iter(())
        return self.table.itemsByIds(self.ids)

    @property
    def _shadow(self) -> "ShadowSelectionManager":
        return _shadow_manager.selection_manager

    @property
    @_experimental()
    def _changed(self) -> Signal:
        """
        Была произведена модификация набора изменяемых объектов.

        Пример::

            axipy._target_object_manager._changed.connect(lambda: print('target changed'))

        :rtype: Signal[]
        """
        return self._shadow.targetChanged
