from typing import (
    TYPE_CHECKING,
    Any,
    ItemsView,
    Iterator,
    KeysView,
    List,
    Mapping,
    Optional,
    ValuesView,
    cast,
)

import axipy
from axipy._internal._mapping_view import (
    ItemsViewExtended,
    KeysViewExtended,
    ValuesViewExtended,
)
from axipy._internal._metaclass2 import _decorate, _MappingMetaExtended
from axipy._internal._utils import _AxiRepr, _Singleton
from PySide2.QtGui import QIcon

from .gui_class import gui_instance

if TYPE_CHECKING:
    from PySide2.QtWidgets import QAction

__all__: List[str] = [
    "ActionManager",
]


def _iter() -> Iterator[str]:
    return map(lambda action: action.objectName(), gui_instance._shadow.all_actions())


def _len() -> int:
    return len(gui_instance._shadow.all_actions())


def _get_item(key: str) -> "QAction":
    result = gui_instance._shadow.find_action_by_id(key)
    if result is None:
        raise KeyError(f"{key}")
    else:
        return result


def _filter_missing(icon_name: str) -> bool:
    try:
        result = QIcon.fromTheme(icon_name)
        if result.isNull():
            return False
    except Exception:
        return False
    return True


class _IconsMapping(Mapping):

    @staticmethod
    def _get_names() -> List[str]:
        return list(filter(_filter_missing, gui_instance._shadow.all_icon_names()))

    def __getitem__(self, key: str) -> QIcon:
        result = QIcon.fromTheme(key)
        if result.isNull():
            raise KeyError(f"{key}")
        else:
            return result

    def __len__(self) -> int:
        return len(self._get_names())

    def __iter__(self) -> Iterator[str]:
        return iter(self._get_names())


class _CustomMappingMetaExtended(_MappingMetaExtended):

    def __iter__(cls) -> Iterator[str]:
        return _iter()

    def __len__(cls) -> int:
        return _len()

    def __getitem__(cls, key: str) -> "QAction":
        return _get_item(key)


class ActionManager(Mapping, _Singleton, _AxiRepr, metaclass=_CustomMappingMetaExtended):
    """
    Менеджер системных действий и инструментов. Класс является словарем, доступным
    только для чтения (:class:`collections.abc.Mapping`), где ключи это идентификаторы
    действий, a значения это объекты класса :class:`PySide2.QtWidgets.QAction`.
    Поддерживает обращение по ключу.

    Note:
        Создание :class:`axipy.ActionManager` не требуется,
        используйте объект :attr:`axipy.action_manager`.

    Вывод перечня доступных действий::

        print(action_manager.keys())

    Получение действия по его наименованию::

        if 'Select' in action_manager.keys():
            action = action_manager['Select']

    Добавление действия в основную панель::

        if 'Select' in action_manager.keys():
            position = Position('Основные', 'Команды')
            button = SystemActionButton('Select')
            position.add(button)

    See also:
        :attr:`MainWindow.quick_toolbar`
    """

    def __new__(cls, *args: Any, **kwargs: Any) -> "ActionManager":
        # back compat
        setattr(cls, "keys", _decorate(cls, cls.keys))
        setattr(cls, "values", _decorate(cls, cls.values))
        setattr(cls, "items", _decorate(cls, cls.items))
        setattr(cls, "get", _decorate(cls, cls.get))

        class _Desc:
            def __get__(self, obj: Any, objtype: Any = None) -> List[str]:
                return list(filter(_filter_missing, gui_instance._shadow.all_icon_names()))

        # back compat
        setattr(cls, "all_icon_names", _Desc())

        def icon_by_name(name: str) -> QIcon:
            return QIcon.fromTheme(name)

        # back compat
        setattr(cls, "icon_by_name", staticmethod(icon_by_name))
        # noinspection PyTypeChecker
        inst = _Singleton.__new__(cls, *args, **kwargs)
        return cast("ActionManager", inst)

    def __len__(self) -> int:
        return _len()

    def __iter__(self) -> Iterator[str]:
        return _iter()

    def __getitem__(self, key: str) -> "QAction":
        return _get_item(key)

    def keys(self) -> KeysView[str]:
        """Возвращает набор ключей, где ключи это идентификаторы действий."""
        return KeysViewExtended(self)

    def values(self) -> "ValuesView[QAction]":
        """Возвращает коллекцию значений, где значения это объекты класса
        :class:`PySide2.QtWidgets.QAction`."""
        return ValuesViewExtended(self)

    def items(self) -> "ItemsView[str, QAction]":
        """Возвращает набор кортежей ключ-значение, где ключи это идентификаторы
        действий, a значения это объекты класса :class:`PySide2.QtWidgets.QAction`."""
        return ItemsViewExtended(self)

    def get(self, key: str, default_value: Optional[Any] = None) -> "Optional[QAction]":
        """Возвращает значение по ключу."""
        return Mapping.get(self, key, default_value)

    def activate(self, name: str) -> None:
        """
        Делает активным инструмент по его идентификатору если это возможно. Если
        действие не найдено, генерируется исключение. Если действие недоступно в
        настоящий момент (неактивно), установка игнорируется.

        Args:
            name: Идентификатор действия

        Raises:
            ValueError: Если действие не найдено.

        Активация инструмента 'Сдвиг'::

            axipy.action_manager.activate('Pan')

        Вызов диалога 'Стиль символа'::

            axipy.action_manager.activate('SymbolStyle')
        """
        act = self.get(name)
        if act:
            if act.isEnabled():
                act.toggle()
                act.trigger()
        else:
            raise ValueError(f"Действие {name} не найдено.")

    @property
    def icons(self) -> Mapping[str, QIcon]:
        """Возвращает словарь, доступный только для чтения
        (:class:`collections.abc.Mapping`), где ключи это идентификаторы иконок, a
        значения это объекты класса :class:`PySide2.QtGui.QICon`."""
        return _IconsMapping()

    def find_by_text(self, text: str) -> Optional["QAction"]:
        """
        Найти действие по его тексту.
        """
        for v in self.values():
            if v.text() == text:
                return v
        return None

    def activate_selection_tool(self) -> None:
        """
        Активирует инструмент "Выбор" (инструмент по умолчанию).
        """
        axipy.ViewTool.reset()
