from abc import abstractmethod
from functools import cached_property
from typing import (
    TYPE_CHECKING,
    Any,
    Iterator,
    List,
    Optional,
    Tuple,
    Type,
    overload,
)

from axipy._internal._decorator import _deprecated_by
from axipy._internal._metaclass import _MappingMetaDocumentation, _MappingMetaExtended
from axipy.cpp_cs import ShadowAreaUnit, ShadowLinearUnit, ShadowUnit

__all__: List[str] = [
    "Unit",
    "LinearUnit",
    "LinearUnits",
    "AreaUnit",
    "AreaUnits",
    "UnitValue",
]


class Unit:
    _shadow_method_name: str

    @classmethod
    @overload
    def _wrap(cls, shadow: ShadowLinearUnit) -> "LinearUnit": ...

    @classmethod
    @overload
    def _wrap(cls, shadow: ShadowAreaUnit) -> "AreaUnit": ...

    @classmethod
    @overload
    def _wrap(cls, shadow: ShadowUnit) -> "Unit": ...

    @classmethod
    def _wrap(cls, shadow: ShadowUnit) -> "Unit":
        inst = cls.__new__(cls)
        inst._shadow = shadow
        return inst

    @classmethod
    def _wrap_name(cls, shadow_method_name: str) -> "Unit":
        inst = cls.__new__(cls)
        inst._shadow_method_name = shadow_method_name
        return inst

    @cached_property
    @abstractmethod
    def _shadow(self) -> ShadowUnit:
        pass

    @property
    def name(self) -> str:
        """Краткое наименование единиц измерения."""
        return self._shadow.name()

    @property
    def localized_name(self) -> str:
        """Локализованное краткое наименование единиц измерения."""
        return self._shadow.localized_name()

    @property
    def description(self) -> str:
        """Краткое описание."""
        return self._shadow.description()

    @property
    def conversion(self) -> float:
        """Коэффициент преобразования в метры."""
        return self._shadow.conversion()

    def to_unit(self, unit: "Unit", value: float = 1) -> float:
        """
        Перевод значения в другие единицы измерения.

        Args:
            unit: Единицы измерения, в которые необходимо перевести значение.
            value: Значение для перевода.
        """
        if type(self) is type(unit):
            return self._shadow.conversion(unit._shadow) * value
        else:
            raise ValueError("Unit type mismatch")

    def __str__(self) -> str:
        return self.name

    def __repr__(self) -> str:
        return f"axipy.{self.__class__.__name__}s.{self.name.replace(' ', '_')}"


class LinearUnit(Unit):
    """
    Линейная единица измерения.

    Используется для работы с координатами объектов или расстояний.

    Note:
        Получить экземпляр можно через класс :class:`axipy.LinearUnits` по соответствующему атрибуту.

    .. literalinclude:: /../../tests/doc_examples/cs/test_example_unit.py
        :caption: Пример создания
        :pyobject: test_run_example_unit_create
        :lines: 2-
        :dedent: 4
    """

    if TYPE_CHECKING:

        @classmethod
        def _wrap_name(cls, shadow_method_name: str) -> "LinearUnit": ...

    @cached_property
    def _shadow(self) -> ShadowLinearUnit:
        return getattr(ShadowLinearUnit, self._shadow_method_name)()

    @staticmethod
    def from_area_unit(area_unit: "AreaUnit") -> Optional["LinearUnit"]:
        """Возвращает единицу измерения расстояния, соответствующую единице измерения
        площадей."""
        if isinstance(area_unit, AreaUnit):
            name = area_unit.name.replace(" ", "_").replace("sq_", "")
            return LinearUnits.get(name)

    def __eq__(self, other: object) -> bool:
        if not isinstance(other, LinearUnit):
            return NotImplemented
        return self.name == other.name


class AreaUnit(Unit):
    """
    Единица измерения площадей.

    Note:
        Получить экземпляр можно через класс :class:`axipy.AreaUnits` по соответствующему атрибуту.

    .. literalinclude:: /../../tests/doc_examples/cs/test_example_unit.py
        :caption: Пример создания
        :pyobject: test_run_example_unit_create
        :lines: 2-
        :dedent: 4
    """

    if TYPE_CHECKING:

        @classmethod
        def _wrap_name(cls, shadow_method_name: str) -> "AreaUnit": ...

    @cached_property
    def _shadow(self) -> ShadowAreaUnit:
        return getattr(ShadowAreaUnit, self._shadow_method_name)()

    @staticmethod
    def from_linear_unit(linear_unit: "LinearUnit") -> Optional["AreaUnit"]:
        """Возвращает единицу измерения площадей, соответствующую единице измерения
        расстояний."""
        if isinstance(linear_unit, LinearUnit):
            name = "sq_" + linear_unit.name.replace(" ", "_")
            return AreaUnits.get(name)

    def __eq__(self, other: object) -> bool:
        if not isinstance(other, AreaUnit):
            return NotImplemented
        return self.name == other.name


class _LinearUnitsMappingMetaExtended(_MappingMetaExtended[str]):

    def __iter__(cls) -> Iterator[str]:
        return iter(LinearUnits._keys())

    def __len__(cls) -> int:
        return len(LinearUnits._keys())

    def __getitem__(cls, key: str) -> LinearUnit:
        try:
            result = getattr(LinearUnits, key)
        except AttributeError:
            raise KeyError(key)
        else:
            return result


class LinearUnits(_MappingMetaDocumentation, metaclass=_LinearUnitsMappingMetaExtended):
    """
    Единицы измерения расстояний.

    Класс является статическим словарем, доступным только для чтения (:class:`collections.abc.Mapping`).
    Поддерживает обращение по индексу.
    """

    km: LinearUnit = LinearUnit._wrap_name("kilometer")
    """Километры."""
    m: LinearUnit = LinearUnit._wrap_name("meter")
    """Метры."""
    mm: LinearUnit = LinearUnit._wrap_name("millimeter")
    """Миллиметры."""
    cm: LinearUnit = LinearUnit._wrap_name("centimeter")
    """Сантиметры."""
    mi: LinearUnit = LinearUnit._wrap_name("mile")
    """Мили."""
    nmi: LinearUnit = LinearUnit._wrap_name("nautical_mile")
    """Морские мили."""
    inch: LinearUnit = LinearUnit._wrap_name("inch")
    """Дюймы."""
    ft: LinearUnit = LinearUnit._wrap_name("foot")
    """Футы."""
    yd: LinearUnit = LinearUnit._wrap_name("yard")
    """Ярды."""
    survey_ft: LinearUnit = LinearUnit._wrap_name("usFoot")
    """Топографические футы."""
    li: LinearUnit = LinearUnit._wrap_name("link")
    """Линки."""
    ch: LinearUnit = LinearUnit._wrap_name("chain")
    """Чейны."""
    rd: LinearUnit = LinearUnit._wrap_name("rod")
    """Роды."""
    degree: LinearUnit = LinearUnit._wrap_name("degree")
    """Градусы."""

    @classmethod
    def _keys(cls) -> Tuple[str, ...]:
        return (
            "km",
            "m",
            "mm",
            "cm",
            "mi",
            "nmi",
            "inch",
            "ft",
            "yd",
            "survey_ft",
            "li",
            "ch",
            "rd",
            "degree",
        )

    @classmethod
    def items(cls) -> List[Tuple[str, LinearUnit]]:
        """Возвращает список кортежей ключ-значение, где ключи это атрибуты класса, а
        значения это объекты класса :class:`axipy.LinearUnit`."""
        return super().items()

    @classmethod
    def keys(cls) -> List[str]:
        """Возвращает список ключей, где ключи это атрибуты класса."""
        return super().keys()

    @classmethod
    def values(cls) -> List[LinearUnit]:
        """Возвращает список значений, где значения это объекты класса
        :class:`axipy.LinearUnit`."""
        return super().values()

    @classmethod
    def get(cls, key: str, default_value: Optional[Any] = None) -> Optional[LinearUnit]:
        """Возвращает значение по ключу."""
        return super().get(key, default_value)


class _AreaUnitsMappingMetaExtended(_MappingMetaExtended[str]):

    def __iter__(cls) -> Iterator[str]:
        return iter(AreaUnits._keys())

    def __len__(cls) -> int:
        return len(AreaUnits._keys())

    def __getitem__(cls, key: str) -> AreaUnit:
        try:
            result = getattr(AreaUnits, key)
        except AttributeError:
            raise KeyError(key)
        else:
            return result


class AreaUnits(_MappingMetaDocumentation, metaclass=_AreaUnitsMappingMetaExtended):
    """
    Единицы измерения площадей.

    Класс является статическим словарем, доступным только для чтения (:class:`collections.abc.Mapping`).
    Поддерживает обращение по индексу.
    """

    sq_mm: AreaUnit = AreaUnit._wrap_name("sq_millimeter")
    """Квадратные миллиметры."""
    sq_cm: AreaUnit = AreaUnit._wrap_name("sq_centimeter")
    """Квадратные сантиметры."""
    sq_m: AreaUnit = AreaUnit._wrap_name("sq_meter")
    """Квадратные метры."""
    sq_km: AreaUnit = AreaUnit._wrap_name("sq_kilometer")
    """Квадратные километры."""
    sq_mi: AreaUnit = AreaUnit._wrap_name("sq_mile")
    """Квадратные мили."""
    sq_nmi: AreaUnit = AreaUnit._wrap_name("sq_nautical_mile")
    """Квадратные морские мили."""
    sq_inch: AreaUnit = AreaUnit._wrap_name("sq_inch")
    """Квадратные дюймы."""
    sq_ft: AreaUnit = AreaUnit._wrap_name("sq_foot")
    """Квадратные футы."""
    sq_yd: AreaUnit = AreaUnit._wrap_name("sq_yard")
    """Квадратные ярды."""
    sq_survey_ft: AreaUnit = AreaUnit._wrap_name("sq_usFoot")
    """Квадратные топографические футы."""
    acre: AreaUnit = AreaUnit._wrap_name("acre")
    """Акры."""
    hectare: AreaUnit = AreaUnit._wrap_name("hectare")
    """Гектары."""
    sq_li: AreaUnit = AreaUnit._wrap_name("sq_link")
    """Квадратные линки."""
    sq_ch: AreaUnit = AreaUnit._wrap_name("sq_chain")
    """Квадратные чейны."""
    sq_rd: AreaUnit = AreaUnit._wrap_name("sq_rod")
    """Квадратные роды."""
    perch: AreaUnit = AreaUnit._wrap_name("perch")
    """Перчи."""
    rood: AreaUnit = AreaUnit._wrap_name("rood")
    """Руды."""

    @classmethod
    def _keys(cls) -> Tuple[str, ...]:
        return (
            "sq_mm",
            "sq_cm",
            "sq_m",
            "sq_km",
            "sq_mi",
            "sq_nmi",
            "sq_inch",
            "sq_ft",
            "sq_yd",
            "sq_survey_ft",
            "acre",
            "hectare",
            "sq_li",
            "sq_ch",
            "sq_rd",
            "perch",
            "rood",
        )

    @classmethod
    def items(cls) -> List[Tuple[str, AreaUnit]]:
        """Возвращает список кортежей ключ-значение, где ключи это атрибуты класса, а
        значения это объекты класса :class:`axipy.AreaUnit`."""
        return super().items()

    @classmethod
    def keys(cls) -> List[str]:
        """Возвращает список ключей, где ключи это атрибуты класса."""
        return super().keys()

    @classmethod
    def values(cls) -> List[AreaUnit]:
        """Возвращает список значений, где значения это объекты класса
        :class:`axipy.AreaUnit`."""
        return super().values()

    @classmethod
    def get(cls, key: str, default_value: Optional[Any] = None) -> Optional[AreaUnit]:
        """Возвращает значение по ключу."""
        return super().get(key, default_value)


class UnitValue:
    """
    Контейнер, который хранит значение вместе с его единицей измерения.

    Пример::

        unit = axipy.UnitValue(2, axipy.LinearUnits.km)
        print(unit)
        >>> "2 km"
    """

    def __init__(self, value: float = 1, unit: Unit = LinearUnits.m):
        """
        Конструктор класса.

        Args:
            value: Значение.
            unit: Единица измерения, в которой содержится значение.
                Если значение не указано, принимаются метры :attr:`LinearUnits.m`.
        """
        self._value: float = value

        if unit is None:  # back compat
            unit = LinearUnits.m

        self._unit: Unit = unit

    @property
    def unit(self) -> Unit:
        """Устанавливает или возвращает единицу измерения."""
        return self._unit

    @unit.setter
    def unit(self, unit: Unit) -> None:
        self._unit = unit

    @property
    def value(self) -> float:
        """Устанавливает или возвращает значение."""
        return self._value

    @value.setter
    def value(self, value: float) -> None:
        self._value = value

    def to_unit(self, unit: Unit) -> "UnitValue":
        new_value = self.unit.to_unit(unit, self.value)
        return UnitValue(new_value, unit)

    def __str__(self) -> str:
        return f"{self.value} {self.unit}"

    def __repr__(self) -> str:
        return f"axipy.{self.__class__.__name__}({self.value!r}, {self.unit!r})"

    # Arithmetic operators

    def __neg__(self) -> "UnitValue":
        """-self"""
        return UnitValue(-self.value, self.unit)

    def __pos__(self) -> "UnitValue":
        """+self."""
        return self

    def __add__(self, other: "UnitValue") -> Optional["UnitValue"]:
        """Self + other."""
        if isinstance(other, UnitValue):
            if self.unit == other.unit:
                return UnitValue(self.value + other.value, self.unit)
            else:
                new_other_value = other.unit.to_unit(self.unit, other.value)
                return UnitValue(self.value + new_other_value, self.unit)

    def __radd__(self, other: "UnitValue") -> Optional["UnitValue"]:
        """Other + self."""
        other, self = self, other  # noqa
        if isinstance(other, UnitValue):
            if self.unit == other.unit:
                return UnitValue(self.value + other.value, self.unit)
            else:
                new_other_value = other.unit.to_unit(self.unit, other.value)
                return UnitValue(self.value + new_other_value, self.unit)

    def __sub__(self, other: "UnitValue") -> Optional["UnitValue"]:
        """self - other"""
        return self + -other

    def __rsub__(self, other: "UnitValue") -> Optional["UnitValue"]:
        """other - self"""
        return -self + other

    def __mul__(self, other: "UnitValue") -> Optional["UnitValue"]:
        """Self * other."""
        if isinstance(other, UnitValue):
            if self.unit == other.unit:
                return UnitValue(self.value * other.value, self.unit)
            else:
                new_other_value = other.unit.to_unit(self.unit, other.value)
                return UnitValue(self.value * new_other_value, self.unit)

    def __rmul__(self, other: "UnitValue") -> Optional["UnitValue"]:
        """Other * self."""
        other, self = self, other  # noqa
        if isinstance(other, UnitValue):
            if self.unit == other.unit:
                return UnitValue(self.value * other.value, self.unit)
            else:
                new_other_value = other.unit.to_unit(self.unit, other.value)
                return UnitValue(self.value * new_other_value, self.unit)

    def __truediv__(self, other: "UnitValue") -> Optional["UnitValue"]:
        """Self / other."""
        if isinstance(other, UnitValue):
            if self.unit == other.unit:
                return UnitValue(self.value / other.value, self.unit)
            else:
                new_other_value = other.unit.to_unit(self.unit, other.value)
                return UnitValue(self.value / new_other_value, self.unit)

    def __rtruediv__(self, other: "UnitValue") -> Optional["UnitValue"]:
        """Other / self."""
        other, self = self, other  # noqa
        if isinstance(other, UnitValue):
            if self.unit == other.unit:
                return UnitValue(self.value / other.value, self.unit)
            else:
                new_other_value = other.unit.to_unit(self.unit, other.value)
                return UnitValue(self.value / new_other_value, self.unit)

    # Comparison

    def __eq__(self, other: object) -> bool:
        if not isinstance(other, UnitValue):
            return NotImplemented

        if self.unit == other.unit:
            return self.value == other.value
        else:
            new_other_value = other.unit.to_unit(self.unit, other.value)
            return self.value == new_other_value

    def __lt__(self, other: "UnitValue") -> bool:
        if isinstance(other, UnitValue):
            if self.unit == other.unit:
                return self.value < other.value
            else:
                new_other_value = other.unit.to_unit(self.unit, other.value)
                return self.value < new_other_value

        return False

    def __le__(self, other: "UnitValue") -> bool:
        if isinstance(other, UnitValue):
            if self.unit == other.unit:
                return self.value <= other.value
            else:
                new_other_value = other.unit.to_unit(self.unit, other.value)
                return self.value <= new_other_value

        return False

    def __gt__(self, other: "UnitValue") -> bool:
        if isinstance(other, UnitValue):
            if self.unit == other.unit:
                return self.value > other.value
            else:
                new_other_value = other.unit.to_unit(self.unit, other.value)
                return self.value > new_other_value

        return False

    def __ge__(self, other: "UnitValue") -> bool:
        if isinstance(other, UnitValue):
            if self.unit == other.unit:
                return self.value >= other.value
            else:
                new_other_value = other.unit.to_unit(self.unit, other.value)
                return self.value >= new_other_value

        return False


def _apply_deprecated() -> None:
    # Linear
    setattr(Unit, "km", LinearUnits.km)
    setattr(Unit, "m", LinearUnits.m)
    setattr(Unit, "mm", LinearUnits.mm)
    setattr(Unit, "cm", LinearUnits.cm)
    setattr(Unit, "mi", LinearUnits.mi)
    setattr(Unit, "nmi", LinearUnits.nmi)
    setattr(Unit, "inch", LinearUnits.inch)
    setattr(Unit, "ft", LinearUnits.ft)
    setattr(Unit, "yd", LinearUnits.yd)
    setattr(Unit, "survey_ft", LinearUnits.survey_ft)
    setattr(Unit, "li", LinearUnits.li)
    setattr(Unit, "ch", LinearUnits.ch)
    setattr(Unit, "rd", LinearUnits.rd)
    setattr(Unit, "degree", LinearUnits.degree)

    # Area
    setattr(Unit, "sq_mm", AreaUnits.sq_mm)
    setattr(Unit, "sq_cm", AreaUnits.sq_cm)
    setattr(Unit, "sq_m", AreaUnits.sq_m)
    setattr(Unit, "sq_km", AreaUnits.sq_km)
    setattr(Unit, "sq_mi", AreaUnits.sq_mi)
    setattr(Unit, "sq_nmi", AreaUnits.sq_nmi)
    setattr(Unit, "sq_inch", AreaUnits.sq_inch)
    setattr(Unit, "sq_ft", AreaUnits.sq_ft)
    setattr(Unit, "sq_yd", AreaUnits.sq_yd)
    setattr(Unit, "sq_survey_ft", AreaUnits.sq_survey_ft)
    setattr(Unit, "acre", AreaUnits.acre)
    setattr(Unit, "hectare", AreaUnits.hectare)
    setattr(Unit, "sq_li", AreaUnits.sq_li)
    setattr(Unit, "sq_ch", AreaUnits.sq_ch)
    setattr(Unit, "sq_rd", AreaUnits.sq_rd)
    setattr(Unit, "perch", AreaUnits.perch)
    setattr(Unit, "rood", AreaUnits.rood)

    class _DescAllLinear:
        def __get__(self, *args: Any, **kwargs: Any) -> List["LinearUnit"]:
            # No degree in deprecated version
            return [getattr(LinearUnit, key) for key in LinearUnits.keys() if key != "degree"]

    class _DescAllArea:
        def __get__(self, *args: Any, **kwargs: Any) -> List["AreaUnit"]:
            return [getattr(AreaUnit, key) for key in AreaUnits.keys()]

    setattr(Unit, "all_linear", _DescAllLinear())
    setattr(Unit, "all_area", _DescAllArea())

    @_deprecated_by("LinearUnits.values()")
    def list_all(cls: Type[LinearUnit]) -> "List[LinearUnit]":
        return cls.all_linear  # type: ignore[attr-defined]

    @_deprecated_by("LinearUnits[name]")
    def by_name(_cls: Type[LinearUnit], name: str) -> Optional["LinearUnit"]:
        list_all_linear = LinearUnit.list_all()  # type: ignore[attr-defined]
        return next(
            filter(
                lambda v: name in [v.localized_name, v.name, v.description],  # type: ignore[arg-type, union-attr]
                list_all_linear,
            ),
            None,
        )

    setattr(LinearUnit, "list_all", classmethod(list_all))
    setattr(LinearUnit, "by_name", classmethod(by_name))

    @_deprecated_by("AreaUnits.values()")  # type: ignore[no-redef]
    def list_all(cls: Type[AreaUnit]) -> "List[AreaUnit]":
        return cls.all_area  # type: ignore[attr-defined]

    @_deprecated_by("AreaUnits[name]")  # type: ignore[no-redef]
    def by_name(_cls: Type[AreaUnit], name: str) -> Optional["AreaUnit"]:
        list_all_area = AreaUnit.list_all()  # type: ignore[attr-defined]
        return next(
            filter(
                lambda v: name in [v.localized_name, v.name, v.description],  # type: ignore[arg-type, union-attr]
                list_all_area,
            ),
            None,
        )

    setattr(AreaUnit, "list_all", classmethod(list_all))
    setattr(AreaUnit, "by_name", classmethod(by_name))


_apply_deprecated()
